/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.scene.control;

import org.sellcom.core.Contract;
import org.sellcom.core.Strings;

import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.scene.control.Menu;
import javafx.scene.control.MenuItem;
import javafx.scene.control.SeparatorMenuItem;
import javafx.scene.input.KeyCombination;

/**
 * Operations with menus.
 *
 * @since 1.0
 */
public class Menus {

	private Menus() {
		// Utility class, not to be instantiated
	}


	/**
	 * Creates a menu.
	 *
	 * @since 1.0
	 */
	public static Menu createMenu(String text) {
		Contract.checkArgument(!Strings.isNullOrEmpty(text), "Text must not be null or empty");

		return new Menu(text);
	}

	/**
	 * Creates a menu item.
	 *
	 * @throws IllegalArgumentException if {@code text} is {@code null} or empty
	 * @throws IllegalArgumentException if {@code actionHandler} is {@code null}
	 *
	 * @since 1.0
	 */
	public static MenuItem createMenuItem(String text, EventHandler<ActionEvent> actionHandler) {
		Contract.checkArgument(!Strings.isNullOrEmpty(text), "Text must not be null or empty");
		Contract.checkArgument(actionHandler != null, "Action handler must not be null");

		MenuItem menuItem = new MenuItem(text);
		menuItem.setOnAction(actionHandler);

		return menuItem;
	}

	/**
	 * Creates a menu item.
	 *
	 * @throws IllegalArgumentException if {@code text} is {@code null} or empty
	 * @throws IllegalArgumentException if {@code accelerator} is {@code null} or empty
	 * @throws IllegalArgumentException if {@code actionHandler} is {@code null}
	 *
	 * @since 1.0
	 */
	public static MenuItem createMenuItem(String text, String accelerator, EventHandler<ActionEvent> actionHandler) {
		Contract.checkArgument(!Strings.isNullOrEmpty(text), "Text must not be null or empty");
		Contract.checkArgument(!Strings.isNullOrEmpty(accelerator), "Accelerator must not be null or empty");
		Contract.checkArgument(actionHandler != null, "Action handler must not be null");

		MenuItem menuItem = new MenuItem(text);
		menuItem.setAccelerator(KeyCombination.keyCombination(accelerator));
		menuItem.setOnAction(actionHandler);

		return menuItem;
	}

	/**
	 * Creates a separator.
	 *
	 * @since 1.0
	 */
	public static SeparatorMenuItem createSeparator() {
		return new SeparatorMenuItem();
	}

}
