/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.scene.control;

import javafx.geometry.Pos;
import javafx.scene.control.TableCell;
import javafx.scene.control.TableColumn;
import javafx.util.Callback;

/**
 * Common {@code TableCellFactory}s.
 *
 * @since 1.0
 */
public class TableCellFactories {

	private TableCellFactories() {
		// Utility class, not to be instantiated
	}


	/**
	 * Returns a factory for center-aligned cells.
	 *
	 * @since 1.0
	 */
	public static <S, T> Callback<TableColumn<S, T>, TableCell<S, T>> centerAlignedCellFactory() {
		return (column) -> new CenterAlignedTableCell<>();
	}

	/**
	 * Returns a factory for right-aligned cells.
	 *
	 * @since 1.0
	 */
	public static <S, T> Callback<TableColumn<S, T>, TableCell<S, T>> rightAlignedCellFactory() {
		return (column) -> new RightAlignedTableCell<>();
	}

	// ------------------------------------------------------------
	// ------------------------------------------------------------
	// ------------------------------------------------------------

	private static class CenterAlignedTableCell<S, T> extends TableCell<S, T> {

		@Override
		protected void updateItem(T item, boolean empty) {
			super.updateItem(item, empty);

			setAlignment(Pos.CENTER);

			if (item != null) {
				setText(item.toString());
			}
		}

	}

	// ------------------------------------------------------------
	// ------------------------------------------------------------
	// ------------------------------------------------------------

	private static class RightAlignedTableCell<S, T> extends TableCell<S, T> {

		@Override
		protected void updateItem(T item, boolean empty) {
			super.updateItem(item, empty);

			setAlignment(Pos.CENTER_RIGHT);

			if (item != null) {
				setText(item.toString());
			}
		}

	}

}
