/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.stage;

import java.io.IOException;

import org.sellcom.core.Contract;
import org.sellcom.core.Strings;

import javafx.fxml.FXMLLoader;
import javafx.scene.Scene;
import javafx.stage.Stage;

/**
 * Base class for {@link Stage}s loaded from FXML resource.
 *
 * @since 1.0
 *
 * @see Stage
 */
public abstract class FxmlStage extends Stage {

	private final Object controller;


	/**
	 * Creates a new stage and loads its contents from the given resource.
	 *
	 * @since 1.0
	 *
	 * @see Class#getResource(String)
	 */
	protected FxmlStage(String resourceName) {
		Contract.checkArgument(!Strings.isNullOrEmpty(resourceName), "Resource name must not be null or empty");

		try {
			// Load FXML
			FXMLLoader loader = new FXMLLoader(getClass().getResource(resourceName));

			Scene scene = new Scene(loader.load());
			scene.getStylesheets().add(getClass().getResource("/resources/styles/validation.css").toExternalForm());
			setScene(scene);

			controller = loader.getController();
		} catch (IOException e) {
			throw new FxmlLoaderException(String.format("Error loading FXML resource: %s", resourceName), e);
		}
	}


	/**
	 * Returns the controller associated with this window.
	 *
	 * @throws IllegalArgumentException if {@code controllerClass} is {@code null}
	 * @throws ClassCastException if the type of the associated controller is not {@code controllerClass}
	 *
	 * @since 1.0
	 */
	@SuppressWarnings("unchecked")
	public <T> T getController(Class<T> controllerClass) {
		Contract.checkArgument(controllerClass != null, "Controller class must not be null");

		return (T) controller;
	}

}
