/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.stage;

import static javafx.collections.FXCollections.observableArrayList;
import static javafx.collections.FXCollections.unmodifiableObservableList;
import static javafx.stage.Screen.getPrimary;
import static javafx.stage.Screen.getScreens;

import java.util.List;
import java.util.stream.Collectors;

import javafx.beans.Observable;
import javafx.beans.property.ReadOnlyListProperty;
import javafx.beans.property.ReadOnlyListWrapper;
import javafx.collections.ObservableList;
import javafx.geometry.Rectangle2D;
import javafx.stage.Screen;

/**
 * Operations with screens.
 *
 * @since 1.0
 */
public class Screens {

	private static final ReadOnlyListWrapper<Screen> additionalScreensWrapper = new ReadOnlyListWrapper<>(observableArrayList());

	static {
		ObservableList<Screen> allScreens = getScreens();
		allScreens.addListener((Observable observable) -> updateScreens());
		updateScreens();
	}


	private Screens() {
		// Utility class, not to be instantiated
	}


	/**
	 * Returns the read-only list property containing the available additional screens.
	 *
	 * @since 1.0
	 */
	public static ReadOnlyListProperty<Screen> additionalScreensProperty() {
		return additionalScreensWrapper.getReadOnlyProperty();
	}

	/**
	 * Returns the available additional screens.
	 *
	 * @since 1.0
	 */
	public static ObservableList<Screen> getAdditionalScreens() {
		return unmodifiableObservableList(additionalScreensWrapper);
	}


	private static void updateScreens() {
		List<Screen> additionalScreens = getScreens().stream()
				.filter(screen -> !screen.equals(getPrimary()))
				.sorted((former, latter) -> {
					int result;

					Rectangle2D formerBounds = former.getBounds();
					Rectangle2D latterBounds = latter.getBounds();

					result = Double.compare(formerBounds.getMinX(), latterBounds.getMinX());
					if (result != 0) {
						return result;
					}
					result = Double.compare(formerBounds.getMinY(), latterBounds.getMinY());
					if (result != 0) {
						return result;
					}

					return 0;
				})
				.collect(Collectors.toList());
		additionalScreensWrapper.setAll(additionalScreens);
	}

}
