/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.application;

import static javafx.geometry.NodeOrientation.LEFT_TO_RIGHT;
import static javafx.geometry.NodeOrientation.RIGHT_TO_LEFT;

import java.nio.file.Path;
import java.util.Locale;

import org.sellcom.core.Contract;
import org.sellcom.core.Strings;
import org.sellcom.core.i18n.Resources;
import org.sellcom.core.io.MorePaths;

import javafx.geometry.NodeOrientation;

/**
 * Base class for JavaFX applications.
 *
 * @since 1.0
 */
public abstract class Application extends javafx.application.Application {

	private static Application INSTANCE;

	static {
		Resources.loadResources("resources/strings/javafx");
	}

	private String appId;

	private static NodeOrientation currentOrientation = getIntrinsicOrientation();


	protected Application() {
		INSTANCE = this;
	}


	/**
	 * Returns the application's identifier.
	 *
	 * @since 1.0
	 */
	public String getAppId() {
		return appId;
	}

	/**
	 * Returns the application's cache directory.
	 *
	 * @since 1.0
	 */
	public Path getCacheDirectory() {
		Contract.checkState(appId != null, "Application identifier has not been set");

		return MorePaths.getUserCacheDirectory(appId);
	}

	/**
	 * Returns the application's config directory.
	 *
	 * @since 1.0
	 */
	public Path getConfigDirectory() {
		Contract.checkState(appId != null, "Application identifier has not been set");

		return MorePaths.getUserConfigDirectory(appId);
	}

	/**
	 * Returns the application's cache directory.
	 *
	 * @since 1.0
	 */
	public Path getDataDirectory() {
		Contract.checkState(appId != null, "Application identifier has not been set");

		return MorePaths.getUserDataDirectory(appId);
	}

	/**
	 * Returns the application instance.
	 *
	 * @since 1.0
	 */
	public static Application getInstance() {
		return INSTANCE;
	}

	/**
	 * Returns the application's log directory.
	 *
	 * @since 1.0
	 */
	public Path getLogDirectory() {
		Contract.checkState(appId != null, "Application identifier has not been set");

		return MorePaths.getUserLogDirectory(appId);
	}

	/**
	 * Returns the application's node orientation.
	 *
	 * @since 1.0
	 */
	public NodeOrientation getNodeOrientation() {
		return currentOrientation;
	}

	/**
	 * Sets the application's identifier.
	 *
	 * @throws IllegalArgumentException if {@code appId} is {@code null} or empty
	 *
	 * @since 1.0
	 */
	public void setAppId(String appId) {
		Contract.checkArgument(!Strings.isNullOrEmpty(appId), "Applicaton identifier must not be null or empty");

		this.appId = appId;
	}


	private static NodeOrientation getIntrinsicOrientation() {
		Locale defaultLocale = Locale.getDefault();

		switch (defaultLocale.getScript()) {
			case "Arab": { // Arabic
				return RIGHT_TO_LEFT;
			}
			case "Hebr": { // Hebrew
				return RIGHT_TO_LEFT;
			}
			case "Thaa": { // Thaana
				return RIGHT_TO_LEFT;
			}
		}

		switch (defaultLocale.getLanguage()) {
			case "ar": { // Arabic
				return RIGHT_TO_LEFT;
			}
			case "dv": { // Maldivian
				return RIGHT_TO_LEFT;
			}
			case "fa": { // Persian
				return RIGHT_TO_LEFT;
			}
			case "iw": { // Hebrew
				return RIGHT_TO_LEFT;
			}
			case "ps": { // Pashto
				return RIGHT_TO_LEFT;
			}
			case "ur": { // Urdu
				return RIGHT_TO_LEFT;
			}
			case "ji": { // Yiddish
				return RIGHT_TO_LEFT;
			}
		}

		return LEFT_TO_RIGHT;
	}

}
