/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.collection;

import static javafx.collections.FXCollections.observableSet;

import java.util.EnumSet;

import org.sellcom.core.Contract;

import javafx.collections.ObservableSet;

/**
 * Operations with {@link ObservableSet}s of enum types.
 *
 * @since 1.0
 *
 * @see EnumSet
 * @see ObservableSet
 */
public class ObservableEnumSet {

	private ObservableEnumSet() {
		// Utility class, not to be instantiated
	}


	/**
	 * Creates an observable set containing all the elements in the given element type.
	 *
	 * @throws IllegalArgumentException if {@code elementType} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see EnumSet#allOf(Class)
	 */
	public static <E extends Enum<E>> ObservableSet<E> allOf(Class<E> elementType) {
		Contract.checkArgument(elementType != null, "Element type must not be null");

		return observableSet(EnumSet.allOf(elementType));
	}

	/**
	 * Creates an observable set containing all the elements not contained in the given enum set.
	 *
	 * @throws IllegalArgumentException if {@code enumSet} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see EnumSet#complementOf(EnumSet)
	 */
	public static <E extends Enum<E>> ObservableSet<E> complementOf(EnumSet<E> enumSet) {
		Contract.checkArgument(enumSet != null, "Enum set must not be null");

		return observableSet(EnumSet.complementOf(enumSet));
	}

	/**
	 * Creates an empty observable set with the given element type.
	 *
	 * @throws IllegalArgumentException if {@code elementType} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see EnumSet#noneOf(Class)
	 */
	public static <E extends Enum<E>> ObservableSet<E> noneOf(Class<E> elementType) {
		Contract.checkArgument(elementType != null, "Element type must not be null");

		return observableSet(EnumSet.noneOf(elementType));
	}

	/**
	 * Creates an observable set containing the given elements.
	 *
	 * @throws IllegalArgumentException if {@code firstElement} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see EnumSet#of(Enum, Enum...)
	 */
	@SafeVarargs
	public static <E extends Enum<E>> ObservableSet<E> of(E firstElement, E... otherElements) {
		Contract.checkArgument(firstElement != null, "First element set must not be null");

		return observableSet(EnumSet.of(firstElement, otherElements));
	}

	/**
	 * Creates an observable set containing all the elements in the given range.
	 * Both the first element and the last element are inclusive.
	 *
	 * @throws IllegalArgumentException if {@code firstElement} is {@code null}
	 * @throws IllegalArgumentException if {@code lastElement} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see EnumSet#range(Enum, Enum)
	 */
	public static <E extends Enum<E>> ObservableSet<E> range(E firstElement, E lastElement) {
		Contract.checkArgument(firstElement != null, "First element set must not be null");
		Contract.checkArgument(lastElement != null, "Last element set must not be null");

		return observableSet(EnumSet.range(firstElement, lastElement));
	}

}
