/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.concurrent;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.sellcom.core.Contract;

import javafx.concurrent.Task;

/**
 * Operations with {@code Runnable}s and {@code Task}s.
 *
 * @since 1.0
 *
 * @see Task
 */
public class Tasks {

	private static final int SCHEDULED_TASK_THREAD_PRIORITY = Thread.NORM_PRIORITY - 2;

	private static final int SIMPLE_TASK_THREAD_PRIORITY = Thread.NORM_PRIORITY;

	private static final ScheduledExecutorService scheduledTaskExecutor = Executors.newSingleThreadScheduledExecutor(Tasks::createScheduledTaskThread);

	private static final AtomicInteger scheduledTaskThreadCounter = new AtomicInteger(0);

	private static final ExecutorService simpleTaskExecutor = Executors.newCachedThreadPool(Tasks::createSimpleTaskThread);

	private static final AtomicInteger simpleTaskThreadCounter = new AtomicInteger(0);


	private Tasks() {
		// Utility class, not to be instantiated
	}


	/**
	 * Schedules the given background task to be executed once with the given delay.
	 *
	 * @throws IllegalArgumentException if {@code task} is {@code null}
	 * @throws IllegalArgumentException if {@code delay} is negative
	 * @throws IllegalArgumentException if {@code unit} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see ScheduledExecutorService#schedule(Runnable, long, TimeUnit)
	 */
	public static <T> Future<T> schedule(Task<T> task, long delay, TimeUnit unit) {
		Contract.checkArgument(task != null, "Task must not be null");
		Contract.checkArgument(delay >= 0, "Delay must not be negative");
		Contract.checkArgument(unit != null, "Time unit must not be null");

		scheduledTaskExecutor.schedule(task, delay, unit);

		return task;
	}

	/**
	 * Schedules the given background task to be executed periodically at the given rate.
	 *
	 * @throws IllegalArgumentException if {@code task} is {@code null}
	 * @throws IllegalArgumentException if {@code initialDelay} is negative
	 * @throws IllegalArgumentException if {@code period} is negative
	 * @throws IllegalArgumentException if {@code unit} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see ScheduledExecutorService#scheduleAtFixedRate(Runnable, long, long, TimeUnit)
	 */
	public static <T> Future<T> scheduleAtFixedRate(Task<T> task, long initialDelay, long period, TimeUnit unit) {
		Contract.checkArgument(task != null, "Task must not be null");
		Contract.checkArgument(initialDelay >= 0, "Initial delay must not be negative");
		Contract.checkArgument(period >= 0, "Period must not be negative");
		Contract.checkArgument(unit != null, "Time unit must not be null");

		scheduledTaskExecutor.scheduleAtFixedRate(task, initialDelay, period, unit);

		return task;
	}

	/**
	 * Schedules the given background task to be executed periodically at the given rate.
	 *
	 * @throws IllegalArgumentException if {@code task} is {@code null}
	 * @throws IllegalArgumentException if {@code initialDelay} is negative
	 * @throws IllegalArgumentException if {@code delay} is negative
	 * @throws IllegalArgumentException if {@code unit} is {@code null}
	 *
	 * @since 1.0
	 *
	 * @see ScheduledExecutorService#scheduleWithFixedDelay(Runnable, long, long, TimeUnit)
	 */
	public static <T> Future<T> scheduleWithFixedDelay(Task<T> task, long initialDelay, long delay, TimeUnit unit) {
		Contract.checkArgument(task != null, "Task must not be null");
		Contract.checkArgument(initialDelay >= 0, "Initial delay must not be negative");
		Contract.checkArgument(delay >= 0, "Delay must not be negative");
		Contract.checkArgument(unit != null, "Time unit must not be null");

		scheduledTaskExecutor.scheduleWithFixedDelay(task, initialDelay, delay, unit);

		return task;
	}

	/**
	 * Submits the given background task to be executed once as soon as possible.
	 *
	 * @throws IllegalArgumentException if {@code task} is {@code null}
	 *
	 * @since 1.0
	 */
	public static <T> Future<T> submit(Task<T> task) {
		Contract.checkArgument(task != null, "Task must not be null");

		simpleTaskExecutor.submit(task);

		return task;
	}


	private static Thread createScheduledTaskThread(Runnable runnable) {
		Thread thread = new Thread(runnable);
		thread.setDaemon(true);
		thread.setName("ScheduledBackgroundTaskThread-" + scheduledTaskThreadCounter.incrementAndGet());
		thread.setPriority(SCHEDULED_TASK_THREAD_PRIORITY);

		return thread;
	}

	private static Thread createSimpleTaskThread(Runnable runnable) {
		Thread thread = new Thread(runnable);
		thread.setDaemon(true);
		thread.setName("SimpleBackgroundTaskThread-" + simpleTaskThreadCounter.incrementAndGet());
		thread.setPriority(SIMPLE_TASK_THREAD_PRIORITY);

		return thread;
	}

}
