/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.geometry;

import static javafx.geometry.HPos.LEFT;
import static javafx.geometry.HPos.RIGHT;
import static javafx.geometry.NodeOrientation.INHERIT;

import org.sellcom.core.Contract;

import javafx.geometry.HPos;
import javafx.geometry.NodeOrientation;

/**
 * Operations with {@code NodeOrientation}s.
 *
 * @since 1.0
 */
public class NodeOrientations {

	private NodeOrientations() {
		// Utility class, not to be instantiated
	}


	/**
	 * Returns the end position for the given node orientation.
	 *
	 * @throws IllegalArgumentException if {@code orientation} is {@code null}
	 * @throws IllegalArgumentException if {@code orientation} is {@code INHERIT}
	 *
	 * @since 1.0
	 */
	public static HPos getEnd(NodeOrientation orientation) {
		Contract.checkArgument(orientation != null, "Orientation must not be null");
		Contract.checkArgument(orientation != INHERIT, "Orientation must not be INHERIT");

		switch (orientation) {
			case LEFT_TO_RIGHT: {
				return RIGHT;
			}
			case RIGHT_TO_LEFT: {
				return LEFT;
			}
			default: {
				throw new AssertionError(String.format("Invalid orientation: %s", orientation));
			}
		}
	}

	/**
	 * Returns the start position for the given node orientation.
	 *
	 * @throws IllegalArgumentException if {@code orientation} is {@code null}
	 * @throws IllegalArgumentException if {@code orientation} is {@code INHERIT}
	 *
	 * @since 1.0
	 */
	public static HPos getStart(NodeOrientation orientation) {
		Contract.checkArgument(orientation != null, "Orientation must not be null");
		Contract.checkArgument(orientation != INHERIT, "Orientation must not be INHERIT");

		switch (orientation) {
			case LEFT_TO_RIGHT: {
				return LEFT;
			}
			case RIGHT_TO_LEFT: {
				return RIGHT;
			}
			default: {
				throw new AssertionError(String.format("Invalid orientation: %s", orientation));
			}
		}
	}

}
