/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.scene.input;

import java.util.EnumSet;
import java.util.Set;

import org.sellcom.core.Contract;

import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.KeyCode;
import javafx.scene.input.KeyEvent;

/**
 * Operations with {@code KeyEvent}s.
 *
 * @since 1.0
 */
public class KeyEvents {

	private static final EventConsumingHandler consumingHandler = new EventConsumingHandler();


	private KeyEvents() {
		// Utility class, not to be instantiated
	}


	/**
	 * Disables the {@code KeyPressed} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void disableKeyPressed(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.addEventFilter(KeyEvent.KEY_PRESSED, consumingHandler);
	}

	/**
	 * Disables the {@code KeyReleased} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void disableKeyReleased(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.addEventFilter(KeyEvent.KEY_RELEASED, consumingHandler);
	}

	/**
	 * Disables the {@code KeyTyped} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void disableKeyTyped(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.addEventFilter(KeyEvent.KEY_TYPED, consumingHandler);
	}

	/**
	 * Enables the {@code KeyPressed} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void enableKeyPressed(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.removeEventFilter(KeyEvent.KEY_PRESSED, consumingHandler);
	}

	/**
	 * Enables the {@code KeyReleased} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void enableKeyReleased(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.removeEventFilter(KeyEvent.KEY_RELEASED, consumingHandler);
	}

	/**
	 * Enables the {@code KeyTyped} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void enableKeyTyped(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.removeEventFilter(KeyEvent.KEY_TYPED, consumingHandler);
	}

	/**
	 * Checks whether the given event is the given main key with no modifier keys.
	 *
	 * @throws IllegalArgumentException if {@code event} is {@code null}
	 * @throws IllegalArgumentException if {@code mainKey} is {@code null}
	 *
	 * @since 1.0
	 */
	public static boolean is(KeyEvent event, KeyCode mainKey) {
		return is(event, mainKey, EnumSet.noneOf(KeyCode.class));
	}

	/**
	 * Checks whether the given event is a combination of the given main key and the given modifier keys.
	 *
	 * @throws IllegalArgumentException if {@code event} is {@code null}
	 * @throws IllegalArgumentException if {@code mainKey} is {@code null}
	 * @throws IllegalArgumentException if {@code modifierKeys} is {@code null}
	 *
	 * @since 1.0
	 */
	public static boolean is(KeyEvent event, KeyCode mainKey, Set<KeyCode> modifierKeys) {
		Contract.checkArgument(event != null, "Event must not be null");
		Contract.checkArgument(mainKey != null, "Main key must not be null");
		Contract.checkArgument(modifierKeys != null, "Modifier keys must not be null");

		// Test main key
		if (event.getCode() != mainKey) {
			return false;
		}

		// Test modifier keys
		if (modifierKeys.contains(KeyCode.ALT) && !event.isAltDown()) {
			return false;
		}
		if (modifierKeys.contains(KeyCode.CONTROL) && !event.isControlDown()) {
			return false;
		}
		if (modifierKeys.contains(KeyCode.META) && !event.isMetaDown()) {
			return false;
		}
		if (modifierKeys.contains(KeyCode.SHIFT) && !event.isShiftDown()) {
			return false;
		}
		if (modifierKeys.contains(KeyCode.SHORTCUT) && !event.isShortcutDown()) {
			return false;
		}

		return true;
	}

	// ------------------------------------------------------------
	// ------------------------------------------------------------
	// ------------------------------------------------------------

	private static class EventConsumingHandler implements EventHandler<KeyEvent> {

		@Override
		public void handle(KeyEvent event) {
			event.consume();
		}

	}

}

