/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.scene.input;

import org.sellcom.core.Contract;

import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;

/**
 * Operations with {@code MouseEvent}s.
 *
 * @since 1.0
 */
public class MouseEvents {

	private static final EventConsumingHandler consumingHandler = new EventConsumingHandler();


	private MouseEvents() {
		// Utility class, not to be instantiated
	}


	/**
	 * Disables the {@code MouseClicked} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void disableMouseClicked(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.addEventFilter(MouseEvent.MOUSE_CLICKED, consumingHandler);
	}

	/**
	 * Disables the {@code MousePressed} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void disableMousePressed(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.addEventFilter(MouseEvent.MOUSE_PRESSED, consumingHandler);
	}

	/**
	 * Disables the {@code MouseReleased} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void disableMouseReleased(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.addEventFilter(MouseEvent.MOUSE_RELEASED, consumingHandler);
	}

	/**
	 * Enables the {@code MouseClicked} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void enableMouseClicked(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.removeEventFilter(MouseEvent.MOUSE_CLICKED, consumingHandler);
	}

	/**
	 * Enables the {@code MousePressed} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void enableMousePressed(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.removeEventFilter(MouseEvent.MOUSE_PRESSED, consumingHandler);
	}

	/**
	 * Enables the {@code MouseReleased} event on the given node.
	 *
	 * @throws IllegalArgumentException if {@code node} is {@code null}
	 *
	 * @since 1.0
	 */
	public static void enableMouseReleased(Node node) {
		Contract.checkArgument(node != null, "Node must not be null");

		node.removeEventFilter(MouseEvent.MOUSE_RELEASED, consumingHandler);
	}

	// ------------------------------------------------------------
	// ------------------------------------------------------------
	// ------------------------------------------------------------

	private static class EventConsumingHandler implements EventHandler<MouseEvent> {

		@Override
		public void handle(MouseEvent event) {
			event.consume();
		}

	}

}
