package org.sentrysoftware.ipmi.core.connection;

/*-
 * ╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲
 * IPMI Java Client
 * ჻჻჻჻჻჻
 * Copyright 2023 Verax Systems, Sentry Software
 * ჻჻჻჻჻჻
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱╲╱
 */

import org.sentrysoftware.ipmi.core.api.async.ConnectionHandle;
import org.sentrysoftware.ipmi.core.api.sol.CipherSuiteSelectionHandler;
import org.sentrysoftware.ipmi.core.api.sync.IpmiConnector;
import org.sentrysoftware.ipmi.core.coding.commands.PrivilegeLevel;
import org.sentrysoftware.ipmi.core.coding.security.CipherSuite;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Manages multiple {@link Session}s.
 */
public class SessionManager {

    private static final Logger logger = LoggerFactory.getLogger(SessionManager.class);

    private static Integer sessionId = 100;

    /**
     * The session ID generated by the {@link SessionManager}.
     * Auto-incremented.
     */
    public static synchronized int generateSessionId() {
        sessionId %= (Integer.MAX_VALUE / 4);
        return sessionId++;
    }

    public static Session establishSession(IpmiConnector connector, String remoteHost, int remotePort, String user, String password,
                                           CipherSuiteSelectionHandler cipherSuiteSelectionHandler) throws SessionException {
        ConnectionHandle handle = null;

        try {
            handle = connector.createConnection(InetAddress.getByName(remoteHost), remotePort);

            List<CipherSuite> availableCipherSuites = connector.getAvailableCipherSuites(handle);
            CipherSuite cipherSuite = cipherSuiteSelectionHandler.choose(availableCipherSuites);

            if (cipherSuite == null) {
                cipherSuite = CipherSuite.getEmpty();
            }

            connector.getChannelAuthenticationCapabilities(handle, cipherSuite, PrivilegeLevel.Administrator);

            return connector.openSession(handle, user, password, null);
        } catch (Exception e) {
            closeConnection(connector, handle);

            throw new SessionException("Cannot create new session due to exception", e);
        }
    }

    private static void closeConnection(IpmiConnector connector, ConnectionHandle handle) {
        try {
            if (connector != null && handle != null) {
                connector.closeSession(handle);
                connector.tearDown();
            }
        } catch (Exception e) {
            logger.error("Cannot close connection after exception thrown during session establishment.", e);
        }
    }

    private final ConcurrentHashMap<Integer, Session> sessionsPerConnectionHandle;

    public SessionManager() {
        this.sessionsPerConnectionHandle = new ConcurrentHashMap<Integer, Session>();
    }

    /**
     * Either returns the {@link Session} registered for given connection, or creates new IPMI {@link Session}
     * and registers it for the given connection handle, if no session was found for the connection.
     *
     * @param connectionHandle
     *          handle of the connection, for which this session should be registered.
     * @return newly created session object or session already registered for the connection
     */
    public Session registerSession(int sessionId, ConnectionHandle connectionHandle) {
        if (connectionHandle.getUser() == null || connectionHandle.getRemoteAddress() == null) {
            throw new IllegalArgumentException("Given connection handle is incomplete (lacks user or remote address)");
        }

        Session newSession = new Session(sessionId, connectionHandle);
        Session currentSession = sessionsPerConnectionHandle.putIfAbsent(connectionHandle.getHandle(), newSession);

        return currentSession != null ? currentSession : newSession;
    }

    /**
     * Unregisters session for given connection handle.
     *
     * @param connectionHandle
     *          handle of the connection, for which the session should be unregistered.
     */
    public void unregisterSession(ConnectionHandle connectionHandle) {
        sessionsPerConnectionHandle.remove(connectionHandle.getHandle());
    }

    /**
     * Returns session already bound to given connection handle fulfilling given criteria.
     *
     * @param remoteAddress
     *          IP addres of the managed system
     * @param remotePort
     *          UDP port of the managed system
     * @param user
     *          IPMI user for whom the connection is established
     * @return session object fulfilling given criteria, or null if no session was registered for such connection.
     */
    public Session getSessionForCriteria(InetAddress remoteAddress, int remotePort, String user) {
        for (Session session : sessionsPerConnectionHandle.values()) {
            ConnectionHandle sessionConnectionHandle = session.getConnectionHandle();

            if (sessionConnectionHandle.getUser().equals(user)
                    && sessionConnectionHandle.getRemoteAddress().equals(remoteAddress)
                    && sessionConnectionHandle.getRemotePort() == remotePort) {
                return session;
            }
        }

        return null;
    }
}
