/*
 * Decompiled with CFR 0.152.
 */
package org.sentrysoftware.ipmi.client;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.DoubleFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.sentrysoftware.ipmi.client.Utils;
import org.sentrysoftware.ipmi.client.model.DeviceDescription;
import org.sentrysoftware.ipmi.client.model.Fru;
import org.sentrysoftware.ipmi.client.model.FruDevice;
import org.sentrysoftware.ipmi.client.model.Sensor;
import org.sentrysoftware.ipmi.core.coding.commands.chassis.GetChassisStatusResponseData;
import org.sentrysoftware.ipmi.core.coding.commands.fru.record.BoardInfo;
import org.sentrysoftware.ipmi.core.coding.commands.fru.record.FruRecord;
import org.sentrysoftware.ipmi.core.coding.commands.fru.record.ProductInfo;
import org.sentrysoftware.ipmi.core.coding.commands.sdr.GetSensorReadingResponseData;
import org.sentrysoftware.ipmi.core.coding.commands.sdr.record.EntityId;
import org.sentrysoftware.ipmi.core.coding.commands.sdr.record.FruDeviceLocatorRecord;
import org.sentrysoftware.ipmi.core.coding.commands.sdr.record.FullSensorRecord;
import org.sentrysoftware.ipmi.core.coding.commands.sdr.record.SensorUnit;

public class IpmiResultConverter {
    private static final double NO_READING = 255.0;
    private static final DoubleFunction<Double> IDENTITY_FUNCTION = v -> v;
    private static final DoubleFunction<Double> VOLTAGE_CONVERSION_FUNCTION = v -> v * 1000.0;
    private static final DoubleFunction<Double> FAHRENHEIT_TO_CELSIUS_CONVERSION_FUNCTION = v -> (v - 32.0) * 0.5556;
    private static final DoubleFunction<Double> KELVIN_TO_CELSIUS_CONVERSION_FUNCTION = v -> v - 273.15;

    private IpmiResultConverter() {
    }

    public static final String convertResult(List<Fru> frus, List<Sensor> sensors) {
        HashMap<String, FruDevice> frusLookup = new HashMap<String, FruDevice>();
        Stream<String> fruRecords = IpmiResultConverter.processFrus(frus, frusLookup);
        Stream<String> sensorStates = IpmiResultConverter.processSensorStates(sensors, frusLookup);
        Stream<String> sensorReadings = IpmiResultConverter.processSensorReadings(sensors);
        return Stream.of(fruRecords, sensorStates, sensorReadings).flatMap(Function.identity()).collect(Collectors.joining("\n"));
    }

    private static Stream<String> processSensorReadings(List<Sensor> sensors) {
        return sensors.stream().filter(Sensor::isFull).filter(sensor -> sensor.getData() != null && sensor.getRecord() != null).map(IpmiResultConverter::extractFullSensorReadingValue).filter(Objects::nonNull);
    }

    private static String extractFullSensorReadingValue(Sensor fullSensor) {
        GetSensorReadingResponseData data = fullSensor.getData();
        FullSensorRecord fullRecord = (FullSensorRecord)fullSensor.getRecord();
        String sensorId = String.format("%04x", fullRecord.getId());
        String deviceType = DeviceDescription.getDeviceType(fullRecord.getEntityId());
        byte deviceId = fullRecord.getEntityInstanceNumber();
        String sensorName = fullRecord.getName();
        SensorUnit unit = fullRecord.getSensorBaseUnit();
        if (data.getPlainSensorReading() == 255.0 || deviceType == null || unit == null || sensorName == null) {
            return null;
        }
        double value = data.getSensorReading(fullRecord);
        String sensorUniqueId = IpmiResultConverter.buildDeviceUniqueId(deviceType, deviceId);
        switch (unit) {
            case DegreesC: 
            case DegreesF: 
            case DegreesK: {
                return IpmiResultConverter.temperatureRow(fullRecord, sensorId, sensorName, value, unit, sensorUniqueId);
            }
            case Volts: {
                return IpmiResultConverter.voltageRow(fullRecord, sensorId, sensorName, value, sensorUniqueId);
            }
            case Rpm: {
                return IpmiResultConverter.fanSpeedRow(fullRecord, sensorId, sensorName, value, sensorUniqueId);
            }
            case Amps: {
                return IpmiResultConverter.currentRow(sensorId, sensorName, value, sensorUniqueId);
            }
            case Watts: {
                return IpmiResultConverter.powerConsumptionRow(sensorId, sensorName, value, sensorUniqueId);
            }
            case Joules: {
                return IpmiResultConverter.energyRow(sensorId, sensorName, value, sensorUniqueId);
            }
        }
        return null;
    }

    private static String energyRow(String sensorId, String sensorName, double value, String sensorUniqueId) {
        return String.join((CharSequence)";", "Energy", sensorId, sensorName, sensorUniqueId, String.valueOf(value));
    }

    private static String powerConsumptionRow(String sensorId, String sensorName, double value, String sensorUniqueId) {
        return String.join((CharSequence)";", "PowerConsumption", sensorId, sensorName, sensorUniqueId, String.valueOf(value));
    }

    private static String currentRow(String sensorId, String sensorName, double value, String sensorUniqueId) {
        return String.join((CharSequence)";", "Current", sensorId, sensorName, sensorUniqueId, String.valueOf(value));
    }

    private static String fanSpeedRow(FullSensorRecord sensor, String sensorId, String sensorName, double value, String sensorUniqueId) {
        String threshold1 = IpmiResultConverter.getAvailableThreshold(IDENTITY_FUNCTION, sensor.getLowerCriticalThreshold(), sensor.getLowerNonRecoverableThreshold());
        String threshold2 = IpmiResultConverter.getThresholdValue(IDENTITY_FUNCTION, sensor.getLowerNonCriticalThreshold());
        return String.join((CharSequence)";", "Fan", sensorId, sensorName, sensorUniqueId, String.valueOf(value), threshold1, threshold2);
    }

    private static String voltageRow(FullSensorRecord sensor, String sensorId, String sensorName, double value, String sensorUniqueId) {
        String threshold1 = IpmiResultConverter.getAvailableThreshold(VOLTAGE_CONVERSION_FUNCTION, sensor.getLowerNonCriticalThreshold(), sensor.getLowerCriticalThreshold(), sensor.getLowerNonRecoverableThreshold());
        String threshold2 = IpmiResultConverter.getAvailableThreshold(VOLTAGE_CONVERSION_FUNCTION, sensor.getUpperNonCriticalThreshold(), sensor.getUpperCriticalThreshold(), sensor.getUpperNonRecoverableThreshold());
        return String.join((CharSequence)";", "Voltage", sensorId, sensorName, sensorUniqueId, String.valueOf(value * 1000.0), threshold1, threshold2);
    }

    private static String temperatureRow(FullSensorRecord sensor, String sensorId, String sensorName, double value, SensorUnit unit, String sensorUniqueId) {
        DoubleFunction<Double> conversionFunction = IDENTITY_FUNCTION;
        if (SensorUnit.DegreesF.equals((Object)unit)) {
            conversionFunction = FAHRENHEIT_TO_CELSIUS_CONVERSION_FUNCTION;
        } else if (SensorUnit.DegreesK.equals((Object)unit)) {
            conversionFunction = KELVIN_TO_CELSIUS_CONVERSION_FUNCTION;
        }
        String threshold1 = IpmiResultConverter.getThresholdValue(conversionFunction, sensor.getUpperNonCriticalThreshold());
        String threshold2 = IpmiResultConverter.getAvailableThreshold(conversionFunction, sensor.getUpperCriticalThreshold(), sensor.getUpperNonRecoverableThreshold());
        value = conversionFunction.apply(value);
        return String.join((CharSequence)";", "Temperature", sensorId, sensorName, sensorUniqueId, String.valueOf(value), threshold1, threshold2);
    }

    private static Stream<String> processSensorStates(List<Sensor> sensors, Map<String, FruDevice> frusLookup) {
        LinkedHashMap sensorEntries = new LinkedHashMap();
        sensors.forEach(sensor -> IpmiResultConverter.extractSensorStates(sensor, frusLookup, sensorEntries));
        return sensorEntries.values().stream();
    }

    private static void extractSensorStates(Sensor sensor, Map<String, FruDevice> frusLookup, Map<String, String> sensorEntries) {
        String entry;
        String states = sensor.getStates();
        String deviceType = DeviceDescription.getDeviceType(sensor.getEntityId());
        Byte deviceId = sensor.getDeviceId();
        if (Utils.isBlank(states) || states.toLowerCase().contains("=device absent") || deviceId == null || Utils.isBlank(deviceType)) {
            return;
        }
        String deviceUniqueId = IpmiResultConverter.buildDeviceUniqueId(deviceType, deviceId.byteValue());
        if (sensorEntries.containsKey(deviceUniqueId)) {
            entry = sensorEntries.get(deviceUniqueId);
            entry = entry + "|" + states;
        } else {
            FruDevice fru = frusLookup.get(deviceUniqueId);
            String vendor = "";
            String model = "";
            String serialNumber = "";
            if (fru != null) {
                vendor = Utils.getValueOrEmpty(fru.getVendor());
                model = Utils.getValueOrEmpty(fru.getModel());
                serialNumber = Utils.getValueOrEmpty(fru.getSerialNumber());
            }
            entry = String.join((CharSequence)";", deviceType, deviceId.toString(), deviceUniqueId, vendor, model, serialNumber, states);
        }
        sensorEntries.put(deviceUniqueId, entry);
    }

    private static Stream<String> processFrus(List<Fru> frus, Map<String, FruDevice> frusLookup) {
        LinkedList<FruDevice> veryGoodFruList = new LinkedList<FruDevice>();
        LinkedList<FruDevice> goodFruList = new LinkedList<FruDevice>();
        LinkedList<FruDevice> poorFruList = new LinkedList<FruDevice>();
        for (Fru fru : frus) {
            FruDeviceLocatorRecord fruLocator = fru.getFruLocator();
            if (fruLocator == null) continue;
            EntityId entityId = EntityId.parseInt(fruLocator.getFruEntityId());
            String deviceUniqueId = IpmiResultConverter.buildDeviceUniqueId(DeviceDescription.getDeviceType(entityId), fruLocator.getFruEntityInstance());
            ProductInfo productInfo = IpmiResultConverter.getInfo(fru.getFruRecords(), ProductInfo.class);
            FruDevice fruDevice = IpmiResultConverter.hasValidProductInfo(productInfo) ? IpmiResultConverter.createFruDeviceFromProductInfo(veryGoodFruList, goodFruList, poorFruList, entityId, productInfo) : IpmiResultConverter.createFruDeviceFromBoardInfo(poorFruList, fru);
            if (fruDevice == null) continue;
            frusLookup.put(deviceUniqueId, fruDevice);
        }
        return Stream.of(veryGoodFruList, goodFruList, poorFruList).flatMap(Collection::stream).map(FruDevice::toString);
    }

    private static String buildDeviceUniqueId(String deviceType, int instanceId) {
        return String.format("%s %d", deviceType, instanceId);
    }

    private static FruDevice createFruDeviceFromBoardInfo(LinkedList<FruDevice> poorFruList, Fru fru) {
        BoardInfo boardInfo = IpmiResultConverter.getInfo(fru.getFruRecords(), BoardInfo.class);
        if (boardInfo == null) {
            return null;
        }
        String vendor = Utils.getValueOrEmpty(boardInfo.getBoardManufacturer());
        String model = Utils.getValueOrEmpty(boardInfo.getBoardProductName());
        String serialNumber = Utils.getValueOrEmpty(boardInfo.getBoardSerialNumber());
        String partNumber = Utils.getValueOrEmpty(boardInfo.getBoardPartNumber());
        serialNumber = IpmiResultConverter.combineSerialNumberAndPartNumber(serialNumber, partNumber);
        FruDevice fruDevice = null;
        if (!Utils.isEmpty(vendor) || !Utils.isEmpty(model)) {
            fruDevice = new FruDevice(vendor, model, serialNumber);
            poorFruList.addLast(fruDevice);
        }
        return fruDevice;
    }

    private static FruDevice createFruDeviceFromProductInfo(LinkedList<FruDevice> veryGoodFruList, LinkedList<FruDevice> goodFruList, LinkedList<FruDevice> poorFruList, EntityId entityId, ProductInfo productInfo) {
        String vendor = Utils.getValueOrEmpty(productInfo.getManufacturerName());
        String model = Utils.getValueOrEmpty(productInfo.getProductName());
        String serialNumber = Utils.getValueOrEmpty(productInfo.getProductSerialNumber());
        String partNumber = Utils.getValueOrEmpty(productInfo.getProductModelNumber());
        serialNumber = IpmiResultConverter.combineSerialNumberAndPartNumber(serialNumber, partNumber);
        FruDevice fruDevice = new FruDevice(vendor, model, serialNumber);
        if (!Utils.isEmpty(model) && !Utils.isEmpty(serialNumber)) {
            if (IpmiResultConverter.isSystemFru(entityId)) {
                veryGoodFruList.addLast(fruDevice);
            } else if (IpmiResultConverter.isFruPanel(entityId)) {
                goodFruList.addFirst(fruDevice);
            } else {
                goodFruList.addLast(fruDevice);
            }
        } else {
            poorFruList.addLast(fruDevice);
        }
        return fruDevice;
    }

    private static String combineSerialNumberAndPartNumber(String serialNumber, String partNumber) {
        if (Utils.isEmpty(serialNumber) && !Utils.isEmpty(partNumber)) {
            serialNumber = partNumber;
        } else if (!Utils.isEmpty(serialNumber) && !Utils.isEmpty(partNumber)) {
            serialNumber = String.format("%s - %s", serialNumber, partNumber);
        }
        return serialNumber;
    }

    private static boolean isFruPanel(EntityId entityId) {
        return EntityId.FrontPanelBoard.equals((Object)entityId) || EntityId.BackPanelBoard.equals((Object)entityId) || EntityId.ChassisBackPanelBoard.equals((Object)entityId);
    }

    private static boolean isSystemFru(EntityId entityId) {
        return EntityId.SystemChassis.equals((Object)entityId) || EntityId.SystemBoard.equals((Object)entityId);
    }

    private static boolean hasValidProductInfo(ProductInfo productInfo) {
        return productInfo != null && !Utils.isBlank(productInfo.getManufacturerName());
    }

    private static <T extends FruRecord> T getInfo(List<FruRecord> fruRecords, Class<T> clazz) {
        if (fruRecords == null) {
            return null;
        }
        return (T)((FruRecord)fruRecords.stream().filter(clazz::isInstance).map(clazz::cast).findFirst().orElse(null));
    }

    private static String getThresholdValue(DoubleFunction<Double> conversionFunction, double threshold) {
        return threshold != 0.0 ? String.valueOf(Math.round(conversionFunction.apply(threshold))) : "";
    }

    private static String getAvailableThreshold(DoubleFunction<Double> conversionFunction, double ... thresholds) {
        return Arrays.stream(thresholds).filter(threshold -> threshold != 0.0).mapToObj(threshold -> IpmiResultConverter.getThresholdValue(conversionFunction, threshold)).findFirst().orElse("");
    }

    public static String convertResult(GetChassisStatusResponseData chassisStatus) {
        if (chassisStatus == null) {
            return null;
        }
        return "System power state is " + (chassisStatus.isPowerOn() ? "up" : "down");
    }
}

