/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Reflection Util
 *
 * @author Leonard Woo
 */
public class ReflectionUtil {

  private static final Class<?>[] EMPTY_CLASS_ARRAY = new Class<?>[0];

  private static final Object[] EMPTY_OBJECT_ARRAY = new Object[0];

  public static <T> T newInstance(Class<T> clazz)
      throws NoSuchMethodException, IllegalAccessException, InvocationTargetException,
      InstantiationException {
    return clazz.getDeclaredConstructor().newInstance();
  }

  public static <T> T newInstance(Class<T> clazz, Class<?>[] parameterTypes, Object[] initargs)
      throws NoSuchMethodException, IllegalAccessException, InvocationTargetException,
      InstantiationException {
    return clazz.getDeclaredConstructor(parameterTypes).newInstance(initargs);
  }

  public static <T> Method findMethod(Class<T> clazz, String methodName, Class<?>... parameterTypes)
      throws NoSuchMethodException, SecurityException {
    return clazz.getDeclaredMethod(methodName, parameterTypes);
  }

  public static <T> Method findMethod(Class<T> clazz, String methodName)
      throws NoSuchMethodException, SecurityException {
    return findMethod(clazz, methodName, EMPTY_CLASS_ARRAY);
  }

  public static Object invokeMethod(Method method, Object target, Object... args)
      throws InvocationTargetException, IllegalAccessException, IllegalArgumentException {
      return method.invoke(target, args);
  }

  public static Object invokeMethod(Method method, Object target)
      throws InvocationTargetException, IllegalAccessException, IllegalArgumentException {
    return invokeMethod(method, target, EMPTY_OBJECT_ARRAY);
  }

  public static Method[] getDeclaredMethods(Class<?> clazz) {
    Method[] declaredMethods = clazz.getDeclaredMethods();
    return declaredMethods.length == 0? declaredMethods: declaredMethods.clone();
  }

  @SuppressWarnings("unchecked")
  public static <T> T getField(Field field, T target)
    throws IllegalArgumentException, IllegalAccessException {
    return (T) field.get(target);
  }

  public static <T> void setField(Field field, T target, T value)
      throws IllegalArgumentException, IllegalAccessException {
    field.set(target, value);
  }
}
