/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.logging;

import java.lang.reflect.InvocationTargetException;
import org.seppiko.commons.utils.ReflectionUtil;
import org.seppiko.commons.utils.StringUtil;

/**
 * Virtual Logger Manager
 *
 * @author Leonard Woo
 */
public class VLoggerManager {

  private static VLoggerManager INSTANCE = new VLoggerManager();;

  /**
   * Singleton Object
   *
   * @return object
   */
  public static VLoggerManager getInstance() {
    return INSTANCE;
  }

  private VLoggerManager() {
  }

  private String clazzName = VLoggerImpl.class.getName();

  /**
   * Injection {@link VLogger} implement name
   *
   * @param clazzName implement class name
   * @return self object
   */
  public VLoggerManager setClazzName(String clazzName) {
    this.clazzName = clazzName;
    return this;
  }

  /**
   * Get VLogger implement class object,default is {@link VLoggerImpl}
   *
   * @param name logger name
   * @return VLogger
   */
  public VLogger getLogger(String name) {
    try {
      Class<? extends VLogger> clazz = (Class<? extends VLogger>) Class.forName(clazzName);
      return ReflectionUtil.newInstance(clazz, new Class[]{String.class}, new String[]{name});
    } catch (ClassNotFoundException | InstantiationException |
        InvocationTargetException | NoSuchMethodException |
        IllegalAccessException ex) {
      System.err.println("Injection failed");
      System.err.println(ex);
      throw new IllegalArgumentException("Injection failed");
    }
  }

  /**
   * Get VLogger implement class object
   *
   * @param clazz logger clazz
   * @return VLogger
   */
  public VLogger getLogger(Class<?> clazz) {
    return getLogger(clazz.getName());
  }

}
