/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.jdbc;

import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.SQLInvalidAuthorizationSpecException;
import java.sql.SQLNonTransientConnectionException;
import java.sql.SQLSyntaxErrorException;
import java.sql.SQLTimeoutException;
import java.sql.SQLTransactionRollbackException;
import java.sql.SQLTransientConnectionException;

/**
 * @author Leonard Woo
 */
public class ExceptionFactory {

  public static SQLException syntaxError(String message) {
    return new ExceptionFactory().create(message, ErrCode.SyntaxError);
  }

  public static SQLException notSupport(String message) {
    return new ExceptionFactory().create(message, ErrCode.FeatureNotSupported);
  }

  public static SQLException timeout(String message) {
    return new ExceptionFactory().create(message, ErrCode.Timeout);
  }

  private SQLException create(String message, ErrCode errorCode) {
    return createException(message, null, errorCode, null);
  }


  private SQLException create(String message, String sqlState, ErrCode errorCode) {
    return createException(message, sqlState, errorCode, null);
  }

  private SQLException create(String message, ErrCode errorCode, Exception cause) {
    return createException(message, null, errorCode, cause);
  }

  private SQLException createException(String message, String sqlState, ErrCode errorCode, Exception cause) {
    switch (errorCode) {
      case FeatureNotSupported:
        return new SQLFeatureNotSupportedException(message, sqlState, cause);
      case SyntaxError:
        return new SQLSyntaxErrorException(message, sqlState, cause);
      case InvalidAuthorizationSpec:
        return new SQLInvalidAuthorizationSpecException(message, sqlState, cause);
      case IntegrityConstraintViolation:
        return new SQLIntegrityConstraintViolationException(message, sqlState, cause);
      case NonTransientConnection:
        return new SQLNonTransientConnectionException(message, sqlState, cause);
      case TransientConnection:
        return new SQLTransientConnectionException(message, sqlState, cause);
      case TransactionRollback:
        return new SQLTransactionRollbackException(message, sqlState, cause);
      case Timeout:
        return new SQLTimeoutException(message, sqlState, cause);
    }
    return new SQLException(message, sqlState, cause);
  }

  public enum ErrCode {
    FeatureNotSupported(1),
    SyntaxError(2),
    InvalidAuthorizationSpec(3),
    IntegrityConstraintViolation(4),
    NonTransientConnection(5),
    TransientConnection(6),
    TransactionRollback(7),
    Timeout(8),
    ;

    final int code;
    ErrCode(int code) {
      this.code = code;
    }

  }
}
