/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.RowIdLifetime;
import java.sql.SQLException;
import java.sql.Types;

/**
 * @author Leonard Woo
 */
public interface CommonDatabaseMetaData extends DatabaseMetaData {

  @Override
  default boolean isCatalogAtStart() throws SQLException {
    return true;
  }

  @Override
  default int getJDBCMajorVersion() throws SQLException {
    return 4;
  }

  @Override
  default int getJDBCMinorVersion() throws SQLException {
    return 3;
  }

  @Override
  default String getDriverVersion() throws SQLException {
    return getDriverMajorVersion() + "." + getDriverMinorVersion();
  }

  @Override
  default int getSQLStateType() throws SQLException {
    return sqlStateSQL99;
  }

  @Override
  default RowIdLifetime getRowIdLifetime() throws SQLException {
    return RowIdLifetime.ROWID_UNSUPPORTED;
  }

  @Override
  default boolean dataDefinitionCausesTransactionCommit() throws SQLException {
    return true;
  }

  @Override
  default boolean dataDefinitionIgnoredInTransactions() throws SQLException {
    return false;
  }

  @Override
  default boolean allProceduresAreCallable() throws SQLException {
    return true;
  }

  @Override
  default boolean allTablesAreSelectable() throws SQLException {
    return true;
  }

  @Override
  default boolean autoCommitFailureClosesAllResultSets() throws SQLException {
    return false;
  }

  @Override
  default int getMaxBinaryLiteralLength() throws SQLException {
    return Integer.MAX_VALUE;
  }

  @Override
  default int getMaxCharLiteralLength() throws SQLException {
    return Integer.MAX_VALUE;
  }

  @Override
  default int getMaxColumnsInSelect() throws SQLException {
    return Short.MAX_VALUE;
  }

  @Override
  default int getDefaultTransactionIsolation() throws SQLException {
    return Connection.TRANSACTION_REPEATABLE_READ;
  }

  @Override
  default boolean supportsMixedCaseQuotedIdentifiers() throws SQLException {
    return supportsMixedCaseIdentifiers();
  }

  @Override
  default boolean storesUpperCaseQuotedIdentifiers() throws SQLException {
    return storesUpperCaseIdentifiers();
  }

  @Override
  default boolean storesLowerCaseQuotedIdentifiers() throws SQLException {
    return storesLowerCaseIdentifiers();
  }

  @Override
  default boolean storesMixedCaseQuotedIdentifiers() throws SQLException {
    return storesMixedCaseIdentifiers();
  }

  @Override
  default boolean supportsAlterTableWithAddColumn() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsAlterTableWithDropColumn() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsColumnAliasing() throws SQLException {
    return true;
  }

  @Override
  default boolean nullPlusNonNullIsNull() throws SQLException {
    return true;
  }

  @Override
  default boolean nullsAreSortedHigh() throws SQLException {
    return false;
  }

  @Override
  default boolean nullsAreSortedLow() throws SQLException {
    return true;
  }

  @Override
  default boolean nullsAreSortedAtStart() throws SQLException {
    return false;
  }

  @Override
  default boolean nullsAreSortedAtEnd() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsConvert() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsConvert(int fromType, int toType) throws SQLException {
    switch (fromType) {
      case Types.TINYINT:
      case Types.SMALLINT:
      case Types.INTEGER:
      case Types.BIGINT:
      case Types.REAL:
      case Types.FLOAT:
      case Types.DECIMAL:
      case Types.NUMERIC:
      case Types.DOUBLE:
      case Types.BIT:
      case Types.BOOLEAN:
        switch (toType) {
          case Types.TINYINT:
          case Types.SMALLINT:
          case Types.INTEGER:
          case Types.BIGINT:
          case Types.REAL:
          case Types.FLOAT:
          case Types.DECIMAL:
          case Types.NUMERIC:
          case Types.DOUBLE:
          case Types.BIT:
          case Types.BOOLEAN:
          case Types.CHAR:
          case Types.VARCHAR:
          case Types.LONGVARCHAR:
          case Types.BINARY:
          case Types.VARBINARY:
          case Types.LONGVARBINARY:
            return true;
          default:
            return false;
        }

      case Types.BLOB:
        switch (toType) {
          case Types.BINARY:
          case Types.VARBINARY:
          case Types.LONGVARBINARY:
          case Types.CHAR:
          case Types.VARCHAR:
          case Types.LONGVARCHAR:
          case Types.TINYINT:
          case Types.SMALLINT:
          case Types.INTEGER:
          case Types.BIGINT:
          case Types.REAL:
          case Types.FLOAT:
          case Types.DECIMAL:
          case Types.NUMERIC:
          case Types.DOUBLE:
          case Types.BIT:
          case Types.BOOLEAN:
            return true;
          default:
            return false;
        }

      case Types.CHAR:
      case Types.CLOB:
      case Types.VARCHAR:
      case Types.LONGVARCHAR:
      case Types.BINARY:
      case Types.VARBINARY:
      case Types.LONGVARBINARY:
        switch (toType) {
          case Types.BIT:
          case Types.TINYINT:
          case Types.SMALLINT:
          case Types.INTEGER:
          case Types.BIGINT:
          case Types.FLOAT:
          case Types.REAL:
          case Types.DOUBLE:
          case Types.NUMERIC:
          case Types.DECIMAL:
          case Types.CHAR:
          case Types.VARCHAR:
          case Types.LONGVARCHAR:
          case Types.BINARY:
          case Types.VARBINARY:
          case Types.LONGVARBINARY:
          case Types.DATE:
          case Types.TIME:
          case Types.TIMESTAMP:
          case Types.BLOB:
          case Types.CLOB:
          case Types.BOOLEAN:
          case Types.NCHAR:
          case Types.LONGNVARCHAR:
          case Types.NCLOB:
            return true;
          default:
            return false;
        }

      case Types.DATE:
        switch (toType) {
          case Types.DATE:
          case Types.CHAR:
          case Types.VARCHAR:
          case Types.LONGVARCHAR:
          case Types.BINARY:
          case Types.VARBINARY:
          case Types.LONGVARBINARY:
            return true;

          default:
            return false;
        }

      case Types.TIME:
        switch (toType) {
          case Types.TIME:
          case Types.CHAR:
          case Types.VARCHAR:
          case Types.LONGVARCHAR:
          case Types.BINARY:
          case Types.VARBINARY:
          case Types.LONGVARBINARY:
            return true;
          default:
            return false;
        }

      case Types.TIMESTAMP:
        switch (toType) {
          case Types.TIMESTAMP:
          case Types.CHAR:
          case Types.VARCHAR:
          case Types.LONGVARCHAR:
          case Types.BINARY:
          case Types.VARBINARY:
          case Types.LONGVARBINARY:
          case Types.TIME:
          case Types.DATE:
            return true;
          default:
            return false;
        }
      default:
        return false;
    }
  }

  @Override
  default boolean supportsTableCorrelationNames() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsDifferentTableCorrelationNames() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsExpressionsInOrderBy() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsOrderByUnrelated() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsGroupBy() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsGroupByUnrelated() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsGroupByBeyondSelect() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsLikeEscapeClause() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsMultipleResultSets() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsMultipleTransactions() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsNonNullableColumns() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsMinimumSQLGrammar() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsCoreSQLGrammar() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsExtendedSQLGrammar() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsANSI92EntryLevelSQL() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsANSI92IntermediateSQL() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsANSI92FullSQL() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsIntegrityEnhancementFacility() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsOuterJoins() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsFullOuterJoins() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsLimitedOuterJoins() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsSchemasInDataManipulation() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsSchemasInProcedureCalls() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsSchemasInTableDefinitions() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsSchemasInIndexDefinitions() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsSchemasInPrivilegeDefinitions() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsCatalogsInDataManipulation() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsCatalogsInProcedureCalls() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsCatalogsInTableDefinitions() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsCatalogsInIndexDefinitions() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsCatalogsInPrivilegeDefinitions() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsPositionedDelete() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsPositionedUpdate() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsSelectForUpdate() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsStoredProcedures() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsSubqueriesInComparisons() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsSubqueriesInExists() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsSubqueriesInIns() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsSubqueriesInQuantifieds() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsCorrelatedSubqueries() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsUnion() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsUnionAll() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsOpenCursorsAcrossCommit() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsOpenCursorsAcrossRollback() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsOpenStatementsAcrossCommit() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsOpenStatementsAcrossRollback() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsTransactions() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsTransactionIsolationLevel(int level) throws SQLException {
    switch (level) {
      case Connection.TRANSACTION_READ_UNCOMMITTED:
      case Connection.TRANSACTION_READ_COMMITTED:
      case Connection.TRANSACTION_REPEATABLE_READ:
      case Connection.TRANSACTION_SERIALIZABLE:
        return true;
      default:
        return false;
    }
  }

  @Override
  default boolean supportsDataDefinitionAndDataManipulationTransactions() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsDataManipulationTransactionsOnly() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsResultSetType(int type) throws SQLException {
    return (type == ResultSet.TYPE_SCROLL_INSENSITIVE || type == ResultSet.TYPE_FORWARD_ONLY);
  }

  @Override
  default boolean supportsResultSetConcurrency(int type, int concurrency) throws SQLException {
    return (type == ResultSet.TYPE_SCROLL_INSENSITIVE || type == ResultSet.TYPE_FORWARD_ONLY);
  }

  @Override
  default boolean ownUpdatesAreVisible(int type) throws SQLException {
    return supportsResultSetType(type);
  }

  @Override
  default boolean ownDeletesAreVisible(int type) throws SQLException {
    return supportsResultSetType(type);
  }

  @Override
  default boolean ownInsertsAreVisible(int type) throws SQLException {
    return supportsResultSetType(type);
  }

  @Override
  default boolean supportsBatchUpdates() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsSavepoints() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsGetGeneratedKeys() throws SQLException {
    return true;
  }

  @Override
  default int getResultSetHoldability() throws SQLException {
    return ResultSet.HOLD_CURSORS_OVER_COMMIT;
  }

  @Override
  default boolean supportsResultSetHoldability(int holdability) throws SQLException {
    return holdability == ResultSet.HOLD_CURSORS_OVER_COMMIT;
  }

  @Override
  default boolean supportsStatementPooling() throws SQLException {
    return false;
  }

  @Override
  default boolean supportsStoredFunctionsUsingCallSyntax() throws SQLException {
    return true;
  }

  @Override
  default boolean supportsRefCursors() throws SQLException {
    return false;
  }

  @Override
  default <T> T unwrap(Class<T> iface) throws SQLException {
    if (isWrapperFor(iface)) {
      return iface.cast(this);
    }
    throw new SQLException("The receiver is not a wrapper for " + iface.getName());
  }

  @Override
  default boolean isWrapperFor(Class<?> iface) throws SQLException {
    if (iface == null) {
      return false;
    }
    return iface.isInstance(this);
  }

  @Override
  default long getMaxLogicalLobSize() throws SQLException {
    return Integer.toUnsignedLong(-1); // Unsigned Integer Max Value
  }
}
