/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.jdbc;

import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.SQLInvalidAuthorizationSpecException;
import java.sql.SQLNonTransientConnectionException;
import java.sql.SQLSyntaxErrorException;
import java.sql.SQLTimeoutException;
import java.sql.SQLTransactionRollbackException;
import java.sql.SQLTransientConnectionException;

/**
 * @author Leonard Woo
 */
public class ExceptionFactory {

  public static SQLException syntaxError(String message) {
    return new ExceptionFactory().create(message, "42000");
  }

  public static SQLException notSupport(String message) {
    return new ExceptionFactory().create(message, "0A000");
  }

  public static SQLException timeout(String message) {
    return new ExceptionFactory().create(message, "ZZ000");
  }

  public static SQLException unknown(String message) {
    return new ExceptionFactory().create(message);
  }

  private SQLException create(String message) {
    return createException(message, "HY100", -1, null);
  }

  private SQLException create(String message, String sqlState) {
    return createException(message, sqlState, -1, null);
  }

  private SQLException create(String message, String sqlState, int errorCode) {
    return createException(message, sqlState, errorCode, null);
  }

  private SQLException create(String message, String sqlState, Exception cause) {
    return createException(message, sqlState, -1, cause);
  }

  private SQLException createException(String message, String sqlState, int errorCode, Exception cause) {
    String subClazz = sqlState == null? "42": sqlState.substring(0, 2);
    switch (subClazz) {
      case "0A":
        return new SQLFeatureNotSupportedException(message, sqlState, errorCode, cause);
      case "42":
        return new SQLSyntaxErrorException(message, sqlState, errorCode, cause);
      case "28":
        return new SQLInvalidAuthorizationSpecException(message, sqlState, errorCode, cause);
      case "23":
        return new SQLIntegrityConstraintViolationException(message, sqlState, errorCode, cause);
      case "08":
        return new SQLNonTransientConnectionException(message, sqlState, errorCode, cause);
      case "TC":
        return new SQLTransientConnectionException(message, sqlState, errorCode, cause);
      case "40":
        return new SQLTransactionRollbackException(message, sqlState, errorCode, cause);
      case "HY":
        return new SQLException(message, sqlState, errorCode, cause);
      default:
        return new SQLTimeoutException(message, sqlState, errorCode, cause);
    }
  }

}
