/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.logging;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;

/**
 * Abstract Logging Manager
 *
 * @author Leonard Woo
 */
public class AbstractLoggingManager {

  private static AbstractLoggingManager INSTANCE = new AbstractLoggingManager();

  public static AbstractLoggingManager getInstance() {
    return INSTANCE;
  }

  private AbstractLoggingManager() {
    init();
  }

  private String clazzName;

  private void init() {
    List<AbstractLoggingServiceProvider> providers = findAbstractLoggingServiceProviders();
    Report.submitException(providers.size() > 1, "Class path contains multiple providers.");
    if (!providers.isEmpty()) {
      clazzName = providers.get(0).getAbstractLoggingName();
      Report.submitException( (providers.size() > 1), "Actual provider is of type [" + providers.get(0) + "]");
    } else {
      Report.submitPrint("No AbsLogProviders were found.");
      Report.submitPrint("Default implement with GLF");
    }
  }

  private List<AbstractLoggingServiceProvider> findAbstractLoggingServiceProviders() {
    ServiceLoader<AbstractLoggingServiceProvider> loader = ServiceLoader.load(AbstractLoggingServiceProvider.class);
    List<AbstractLoggingServiceProvider> providers = new ArrayList<>();
    loader.forEach(providers::add);
    return providers;
  }

  /**
   * Get Abstract Logging name
   *
   * @param name logger name
   * @return VLogger
   */
  @SuppressWarnings("unchecked")
  public AbstractLogging getLogger(String name) {
    try {
      return ((Class<? extends AbstractLogging>) Class.forName(clazzName))
          .getDeclaredConstructor(String.class).newInstance(name);
    } catch (ClassNotFoundException | InstantiationException |
        InvocationTargetException | NoSuchMethodException |
        IllegalAccessException ex) {
      Report.submitPrint("Injection failed");
      Report.submitPrint(ex);
      throw new IllegalArgumentException("Injection failed");
    }
  }

  /**
   * Get Abstract Logging name with class object
   *
   * @param clazz logger clazz
   * @return VLogger
   */
  public AbstractLogging getLogger(Class<?> clazz) {
    return getLogger(clazz.getName());
  }
}
