/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.logging.event;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;
import org.seppiko.commons.logging.Logging;
import org.seppiko.commons.logging.LoggingBuilder;
import org.seppiko.commons.logging.utils.Report;

/**
 * Default logging implementation
 *
 * @author Leonard Woo
 */
public class DefaultLogging implements Logging {

  /** The default format to use when formatting dates */
  protected static final String DEFAULT_DATE_TIME_FORMAT = "yyyy/MM/dd HH:mm:ss:SSS zzz";
  private static final DateTimeFormatter dateFormatter;

  static {
    dateFormatter = DateTimeFormatter.ofPattern(DEFAULT_DATE_TIME_FORMAT);
  }

  /** current log level. */
  protected volatile int currentLogLevel;
  /** The short name of this simple log instance */
  private volatile String loggingName;

  /** Default constructor */
  public DefaultLogging() {
  }

  /**
   * constructor with logging name
   *
   * @param loggingName logging name.
   */
  public DefaultLogging(String loggingName) {
    this.loggingName = loggingName;
  }

  /**
   * set delegate logging
   *
   * @param logging delegate logging.
   */
  public void setDelegate(Logging logging) {
    loggingName = logging.getName();
  }

  /** {@inheritDoc} */
  @Override
  public String getName() {
    return loggingName;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isEnable(int level) {
    return level >= currentLogLevel;
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message) {
    log(level, message, null);
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message, Throwable cause) {
    log(level, message, cause, (Supplier<?>) null);
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message, Throwable cause, Supplier<?>... paramSuppliers) {
    List<Supplier<?>> list = Arrays.stream(paramSuppliers).filter(Objects::nonNull).toList();
    Object[] params = new Object[list.size()];
    for (int i = 0; i < list.size(); i++) {
      params[i] = list.get(i).get();
    }
    log(level, message, cause, params);
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message, Throwable cause, Object... params) {
    final StringBuffer buf = new StringBuffer();

    final ZonedDateTime dateTime = ZonedDateTime.now();
    String dateText;
    synchronized (dateFormatter) {
      dateText = dateTime.format(dateFormatter);
    }
    buf.append(dateText);
    buf.append(" ");

    switch (level) {
      case LEVEL_TRACE_INT -> buf.append("[TRACE] ");
      case LEVEL_DEBUG_INT -> buf.append("[DEBUG] ");
      case LEVEL_INFO_INT -> buf.append("[INFO]  ");
      case LEVEL_WARN_INT -> buf.append("[WARN]  ");
      case LEVEL_ERROR_INT -> buf.append("[ERROR] ");
      case LEVEL_FATAL_INT -> buf.append("[FATAL] ");
    }

    buf.append(loggingName);
    buf.append(" - ");
    buf.append(message);

    if (params != null && params.length > 0) {
      buf.append(" <");
      Arrays.stream(params)
          .filter(Objects::nonNull)
          .toList()
          .forEach(
              obj -> {
                String objStr = obj.toString();
                buf.append(objStr);
                buf.append(", ");
              });
      buf.append(">");
    }

    Report.diagno(buf.toString(), cause);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isTraceEnable() {
    return isEnable(LEVEL_TRACE_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void trace(CharSequence message) {
    log(LEVEL_TRACE_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void trace(CharSequence message, Throwable cause) {
    log(LEVEL_TRACE_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void trace(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_TRACE_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atTrace() {
    return new DefaultLoggingBuilder(this, LEVEL_TRACE_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isDebugEnable() {
    return isEnable(LEVEL_DEBUG_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void debug(CharSequence message) {
    log(LEVEL_DEBUG_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void debug(CharSequence message, Throwable cause) {
    log(LEVEL_DEBUG_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void debug(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_DEBUG_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atDebug() {
    return new DefaultLoggingBuilder(this, LEVEL_DEBUG_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isInfoEnable() {
    return isEnable(LEVEL_INFO_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void info(CharSequence message) {
    log(LEVEL_INFO_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void info(CharSequence message, Throwable cause) {
    log(LEVEL_INFO_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void info(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_INFO_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atInfo() {
    return new DefaultLoggingBuilder(this, LEVEL_INFO_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isWarnEnable() {
    return isEnable(LEVEL_WARN_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void warn(CharSequence message) {
    log(LEVEL_WARN_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void warn(CharSequence message, Throwable cause) {
    log(LEVEL_WARN_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void warn(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_WARN_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atWarn() {
    return new DefaultLoggingBuilder(this, LEVEL_WARN_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isErrorEnable() {
    return isEnable(LEVEL_ERROR_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void error(CharSequence message) {
    log(LEVEL_ERROR_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void error(CharSequence message, Throwable cause) {
    log(LEVEL_ERROR_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void error(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_ERROR_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atError() {
    return new DefaultLoggingBuilder(this, LEVEL_ERROR_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isFatalEnable() {
    return isEnable(LEVEL_FATAL_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void fatal(CharSequence message) {
    log(LEVEL_FATAL_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void fatal(CharSequence message, Throwable cause) {
    log(LEVEL_FATAL_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void fatal(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_FATAL_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atFatal() {
    return new DefaultLoggingBuilder(this, LEVEL_FATAL_INT);
  }
}
