/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.StringReader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

/**
 * File and Stream Util
 *
 * @author Leonard Woo
 */
public class StreamUtil {

  /**
   * Find and load file from pathname
   *
   * @param clazz classloader loading point
   * @param path relative path
   * @return file input stream
   * @throws FileNotFoundException file not found
   */
  public static InputStream findFile(Class<?> clazz, String path) throws FileNotFoundException {
    try {
      ClassLoader cl = clazz.getClassLoader();
      InputStream is = cl.getResourceAsStream(path);
      if (is == null) {
        try {
          is = new FileInputStream(
              Objects.requireNonNull(cl.getResource(path)).getPath());
        } catch (NullPointerException ex) {
          is = null;
        }
      }
      if (is == null) {
        is = new FileInputStream(path);
      }
      return is;
    } catch (IOException ex) {
      throw new FileNotFoundException();
    }
  }

  /**
   * Get file stream
   *
   * @param filepath File path
   * @return File input stream
   * @throws FileNotFoundException File not found
   */
  public static InputStream getStream(String filepath)
      throws FileNotFoundException {
    return new BufferedInputStream(new FileInputStream(filepath));
  }

  /**
   * Get file stream
   *
   * @param file File object
   * @return File input stream
   * @throws FileNotFoundException File not found
   */
  public static InputStream getStream(File file) throws FileNotFoundException {
    return new BufferedInputStream(new FileInputStream(file));
  }

  /**
   * Load string to reader
   *
   * @param str String
   * @return Reader
   */
  public static BufferedReader loadString(String str) {
    return new BufferedReader(new StringReader(str));
  }

  /**
   * Load reader from InputStream
   *
   * @param is Inputstream
   * @return Reader
   */
  public static BufferedReader loadReader(InputStream is) {
    return new BufferedReader( new InputStreamReader(is) );
  }

  /**
   * Load inputstram to reader with charset
   *
   * @param is      Inputstream
   * @param charset Charest see{@link StandardCharsets}
   * @return Reader
   */
  public static BufferedReader loadReader(InputStream is, Charset charset) {
    return new BufferedReader(new InputStreamReader(is, charset));
  }

  /**
   * Convert reader to String
   *
   * @param reader Reader
   * @return reader content without newline char
   * @throws IOException read exception
   */
  public static String readerToString(BufferedReader reader) throws IOException {
    StringBuffer sb = new StringBuffer();
    while (reader.ready()) {
      sb.append(reader.readLine());
    }
    return sb.toString();
  }

  /**
   * Write file
   * If not exist create it
   *
   * @param pathname file out path
   * @return file writer
   * @throws IOException file create or file writer exception
   */
  public static Writer writeFile(String pathname) throws IOException {
    return writeFile(new File(pathname));
  }

  /**
   * Write file
   * If not exist create it
   *
   * @param file file object
   * @return file writer
   * @throws IOException file create or file writer exception
   */
  public static Writer writeFile(File file) throws IOException {
    if (!file.exists()) {
      file.createNewFile();
    }
    return new FileWriter(file);
  }

  /**
   * Output to file
   *
   * @param b    bytes
   * @param file File
   * @throws IOException I/O exception
   */
  public static void toFile(byte[] b, File file) throws IOException {
    if (!file.exists()) {
      file.createNewFile();
    }
    OutputStream os = new BufferedOutputStream(new FileOutputStream(file));
    os.write(b);
    os.close();
  }

  /**
   * Copy inputstream to outputstream
   *
   * @param is InputStream
   * @param os OutputStream
   * @throws IOException I/O exception
   */
  public static void copy(InputStream is, OutputStream os) throws IOException {
    is.transferTo(os);
    os.flush();
  }
}
