/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpClient.Version;
import java.net.http.HttpHeaders;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import javax.net.ssl.SSLSession;
import lombok.SneakyThrows;

/**
 * Http Async Response Util with java.net.http
 *
 * HttpRequest see {@link HttpClientUtil}
 *
 * @author Leonard Woo
 */
public class AsyncHttpClientUtil {

  /**
   * Get Async String Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpClientException get async response exception
   */
  public static HttpResponse<String> getAsyncResponseString(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    CompletableFuture<HttpResponse<String>> resp =
        (proxy == null?
            getAsyncResponseRaw(req, BodyHandlers.ofString()):
            getAsyncResponseProxyRaw(req, BodyHandlers.ofString(), proxy) );
    try {
      return httpResponse(resp);
    } catch (InterruptedException | ExecutionException ex) {
      throw new HttpClientException("Response exception", ex);
    }
  }

  /**
   * Get Async byte array Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpClientException get async response exception
   */
  public static HttpResponse<byte[]> getAsyncResponseByteArray(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    CompletableFuture<HttpResponse<byte[]>> resp =
        (proxy == null?
            getAsyncResponseRaw(req, BodyHandlers.ofByteArray()):
            getAsyncResponseProxyRaw(req, BodyHandlers.ofByteArray(), proxy) );
    try {
      return httpResponse(resp);
    } catch (InterruptedException | ExecutionException ex) {
      throw new HttpClientException("Response exception", ex);
    }
  }

  /**
   * Get Async InputStream Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpClientException get async response exception
   */
  public static HttpResponse<InputStream> getAsyncResponseInputStream(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    CompletableFuture<HttpResponse<InputStream>> resp =
        (proxy == null?
            getAsyncResponseRaw(req, BodyHandlers.ofInputStream()):
            getAsyncResponseProxyRaw(req, BodyHandlers.ofInputStream(), proxy) );
    try {
      return httpResponse(resp);
    } catch (InterruptedException | ExecutionException ex) {
      throw new HttpClientException("Response exception", ex);
    }
  }

  private static <T> HttpResponse<T> httpResponse(CompletableFuture<HttpResponse<T>> resp)
      throws InterruptedException, ExecutionException {

    return new HttpResponse<>() {

      @SneakyThrows
      @Override
      public int statusCode() {
        return resp.thenApply(HttpResponse::statusCode).get();
      }

      @SneakyThrows
      @Override
      public HttpRequest request() {
        return resp.thenApply(HttpResponse::request).get();
      }

      @SneakyThrows
      @Override
      public Optional<HttpResponse<T>> previousResponse() {
        return resp.thenApply(HttpResponse::previousResponse).get();
      }

      @SneakyThrows
      @Override
      public HttpHeaders headers() {
        return resp.thenApply(HttpResponse::headers).get();
      }

      @SneakyThrows
      @Override
      public T body() {
        return resp.thenApply(HttpResponse::body).get();
      }

      @SneakyThrows
      @Override
      public Optional<SSLSession> sslSession() {
        return resp.thenApply(HttpResponse::sslSession).get();
      }

      @SneakyThrows
      @Override
      public URI uri() {
        return resp.thenApply(HttpResponse::uri).get();
      }

      @SneakyThrows
      @Override
      public Version version() {
        return resp.thenApply(HttpResponse::version).get();
      }
    };
  }

  private static <T> CompletableFuture<HttpResponse<T>> getAsyncResponseRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler) {
    return HttpClient.newHttpClient().sendAsync(req, responseBodyHandler);
  }

  private static <T> CompletableFuture<HttpResponse<T>> getAsyncResponseProxyRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler, InetSocketAddress proxy) {
    return HttpClient.newBuilder().proxy(ProxySelector.of(proxy)).build().sendAsync(req, responseBodyHandler);
  }
}
