/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;

/**
 * Http Client Util with java.net.http
 *
 * @author Leonard Woo
 */
public class HttpClientUtil {

  /**
   * Get Request object with String
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body string
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   */
  public static HttpRequest getRequest(String url, HttpMethod method, int timeout,
      LinkedHashMap<String, ArrayList<String>> headers, String requestBody)
      throws URISyntaxException {

    String[] headerStringArray =
        (headers != null) ? headerParser(headers).toArray(String[]::new) : new String[] {};

    HttpRequest.BodyPublisher requestBodyPublisher =
        requestBody != null
            ? HttpRequest.BodyPublishers.ofString(requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(new URI(url), method, timeout, headerStringArray, requestBodyPublisher);
  }

  /**
   * Get Request object with inputstream
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body inputstream
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   */
  public static HttpRequest getRequest(String url, HttpMethod method, int timeout,
      LinkedHashMap<String, ArrayList<String>> headers, InputStream requestBody)
      throws URISyntaxException {

    String[] headerStringArray =
        (headers != null) ? headerParser(headers).toArray(String[]::new) : new String[] {};

    HttpRequest.BodyPublisher requestBodyPublisher = requestBody != null
            ? HttpRequest.BodyPublishers.ofInputStream(() -> requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(new URI(url), method, timeout, headerStringArray, requestBodyPublisher);
  }

  /**
   * Get Request object with byte array
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body byte array
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   */
  public static HttpRequest getRequest(String url, HttpMethod method, int timeout,
      LinkedHashMap<String, ArrayList<String>> headers, byte[] requestBody)
      throws URISyntaxException {

    String[] headerStringArray =
        (headers != null) ? headerParser(headers).toArray(String[]::new) : new String[] {};

    HttpRequest.BodyPublisher requestBodyPublisher = requestBody != null
            ? HttpRequest.BodyPublishers.ofByteArray(requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(new URI(url), method, timeout, headerStringArray, requestBodyPublisher);
  }

  /**
   * Get Response with String body
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   */
  public static HttpResponse<String> getResponseString(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    try {
      if (proxy != null) {
        return getResponseProxyRaw(req, BodyHandlers.ofString(), proxy);
      }
      return getResponseRaw(req, BodyHandlers.ofString());
    } catch (IOException | InterruptedException ex) {
      throw new HttpClientException(ex.getMessage());
    }
  }

  /**
   * Get Response with byte array body
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   */
  public static HttpResponse<byte[]> getResponseByteArray(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    try {
      if (proxy != null) {
        return getResponseProxyRaw(req, BodyHandlers.ofByteArray(), proxy);
      }
      return getResponseRaw(req, BodyHandlers.ofByteArray());
    } catch (IOException | InterruptedException ex) {
      throw new HttpClientException(ex.getMessage());
    }
  }

  /**
   * Get Response with inputstream body
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   */
  public static HttpResponse<InputStream> getResponseInputStream(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    try {
      if (proxy != null) {
        return getResponseProxyRaw(req, BodyHandlers.ofInputStream(), proxy);
      }
      return getResponseRaw(req, BodyHandlers.ofInputStream());
    } catch (IOException | InterruptedException ex) {
      throw new HttpClientException(ex.getMessage());
    }
  }

  private static ArrayList<String> headerParser(LinkedHashMap<String, ArrayList<String>> headers) {
    ArrayList<String> headerList = new ArrayList<>();
    headers.forEach( (key, value) -> {
      value.forEach( e -> {
        headerList.add(key + ":" + e);
      });
    });
    return headerList;
  }

  private static HttpRequest getRequestRaw(URI uri, HttpMethod method, int timeout, String[] headers,
      HttpRequest.BodyPublisher requestBody) {
    HttpRequest.Builder builder = HttpRequest.newBuilder().uri(uri).method(method.name(), requestBody);
    if (headers.length > 0) {
      builder = builder.headers(headers);
    }
    return builder.timeout(Duration.ofSeconds(timeout)).build();
  }

  private static <T> HttpResponse<T> getResponseRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler)
      throws IOException, InterruptedException {
    return HttpClient.newHttpClient().send(req, responseBodyHandler);
  }

  private static <T> HttpResponse<T> getResponseProxyRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler, InetSocketAddress proxy)
      throws IOException, InterruptedException {
    return HttpClient.newBuilder().proxy(ProxySelector.of(proxy)).build().send(req, responseBodyHandler);
  }
}
