/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.Comparator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;

/**
 * Math Util
 *
 * @author Leonard Woo
 */
public class MathUtil {

  /**
   * BigDecimal list max number
   *
   * @param nums number list
   * @return maximum
   * @throws NoSuchElementException list is null
   */
  public static BigDecimal maxDecimal(List<BigDecimal> nums) throws NoSuchElementException {
    return nums.stream().filter(Objects::nonNull).max(Comparator.comparing(n -> n)).orElseThrow();
  }

  /**
   * BigInteger list max number
   *
   * @param nums number list
   * @return maximum
   * @throws NoSuchElementException list is null
   */
  public static BigInteger maxInteger(List<BigInteger> nums) throws NoSuchElementException {
    return nums.stream().filter(Objects::nonNull).max(Comparator.comparing(n -> n)).orElseThrow();
  }

  /**
   * BigDecimal list min number
   *
   * @param nums number list
   * @return minimum
   * @throws NoSuchElementException list is null
   */
  public static BigDecimal minDecimal(List<BigDecimal> nums) throws NoSuchElementException {
    return nums.stream().filter(Objects::nonNull).min(Comparator.comparing(n -> n)).orElseThrow();
  }

  /**
   * BigInteger list min number
   *
   * @param nums number list
   * @return minimum
   * @throws NoSuchElementException list is null
   */
  public static BigInteger minInteger(List<BigInteger> nums) throws NoSuchElementException {
    return nums.stream().filter(Objects::nonNull).min(Comparator.comparing(n -> n)).orElseThrow();
  }

  /**
   * BigDecimal list avg number
   *
   * @param nums number list
   * @param roundingMode RoundingMode
   * @return average
   */
  public static BigDecimal avgDecimal(List<BigDecimal> nums, RoundingMode roundingMode) {
    return nums.stream().map(Objects::requireNonNull).reduce(BigDecimal.ZERO, BigDecimal::add)
        .divide(BigDecimal.valueOf(nums.size()), roundingMode);
  }

  /**
   * BigDecimal list avg number
   *
   * @param nums number list
   * @return average double
   * @throws NoSuchElementException list is null
   */
  public static double avgDecimal(List<BigDecimal> nums) throws NoSuchElementException {
    return nums.stream().mapToDouble(BigDecimal::doubleValue).average().orElseThrow();
  }

  /**
   * BigInteger list avg number
   *
   * @param nums number list
   * @param roundingMode RoundingMode
   * @return average
   */
  public static BigDecimal avgInteger(List<BigInteger> nums, RoundingMode roundingMode) {
    BigInteger sum = nums.stream().map(Objects::requireNonNull).reduce(BigInteger.ZERO, BigInteger::add);
    return new BigDecimal(sum).divide(BigDecimal.valueOf(nums.size()), roundingMode);
  }

  /**
   * Double format
   *
   * @param precisionPattern precision pattern, e.g. 0.00
   * @param number double number
   * @return formatted string
   */
  public static String format(String precisionPattern, double number) {
    return new DecimalFormat(precisionPattern).format(number);
  }
}
