/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.StringReader;
import java.io.Writer;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * File and Stream Util
 *
 * @author Leonard Woo
 */
public class StreamUtil {

  /**
   * Find and load file from pathname
   *
   * @param clazz classloader loading point
   * @param pathname relative path
   * @return file input stream if file not found is null
   */
  public static InputStream findFileInputStream(Class<?> clazz, String pathname) {
    try {
      return new FileInputStream( findFile(clazz, pathname) );
    } catch (IOException ignored) {
    }
    return null;
  }

  /**
   * Find and load file from pathname
   *
   * @param clazz classloader loading point
   * @param pathname relative path
   * @return file
   */
  public static File findFile(Class<?> clazz, String pathname) {
    String _pathname = getClassLoaderPath(clazz.getClassLoader(), pathname);
    _pathname = (_pathname == null? pathname: _pathname);
    return new File(pathname);
  }

  private static String getClassLoaderPath(ClassLoader cl, String path) {
    URL url = cl.getResource(path);
    return url == null? null: url.getPath();
  }

  /**
   * Get file stream
   *
   * @param filepath File path
   * @return File input stream if File not found is null
   */
  public static InputStream getStream(String filepath) {
    return getStream(new File(filepath));
  }

  /**
   * Get file stream
   *
   * @param file File object
   * @return File input stream if File not found is null
   */
  public static InputStream getStream(File file) {
    try {
      return new BufferedInputStream(new FileInputStream(file));
    } catch (FileNotFoundException ignored) {
    }
    return null;
  }

  /**
   * Load string to reader
   *
   * @param str String
   * @return Reader
   */
  public static BufferedReader loadString(String str) {
    return new BufferedReader(new StringReader(str));
  }

  /**
   * Load reader from InputStream
   *
   * @param is Inputstream
   * @return Reader
   */
  public static BufferedReader loadReader(InputStream is) {
    return new BufferedReader( new InputStreamReader(is) );
  }

  /**
   * Load inputstram to reader with charset
   *
   * @param is      Inputstream
   * @param charset Charest see{@link StandardCharsets}
   * @return Reader
   */
  public static BufferedReader loadReader(InputStream is, Charset charset) {
    return new BufferedReader(new InputStreamReader(is, charset));
  }

  /**
   * export byte array to output stream
   *
   * @param bytes byte array
   * @param os output stream
   * @throws IOException byte array write failed
   */
  public static void exportOutputStream(byte[] bytes, OutputStream os) throws IOException {
    BufferedOutputStream bos = new BufferedOutputStream(os);
    bos.write(bytes);
    bos.flush();
  }

  /**
   * Convert reader to String
   *
   * @param reader Reader
   * @param ignoreNewLine true - read all content without new line
   * @return reader content without newline char
   * @throws IOException read exception
   */
  public static String readerToString(BufferedReader reader, boolean ignoreNewLine) throws IOException {
    StringBuilder sb = new StringBuilder();
    while (reader.ready()) {
      sb.append(reader.readLine());
      if(!ignoreNewLine) {
        sb.append(Environment.NEW_LINE);
      }
    }
    return sb.toString();
  }

  /**
   * Write file
   * If not exist create it
   *
   * @param pathname file out path
   * @return file writer
   * @throws IOException file create or file writer exception
   */
  public static Writer writeFile(String pathname) throws IOException {
    return writeFile(new File(pathname));
  }

  /**
   * Write file
   * If not exist create it
   *
   * @param file file object
   * @return file writer
   * @throws IOException file create or file writer exception
   */
  public static Writer writeFile(File file) throws IOException {
    if (!file.exists()) {
      if(!file.createNewFile()) {
        throw new IOException("Can not create file");
      }
    }
    return new FileWriter(file);
  }

  /**
   * Output to file
   *
   * @param b    bytes
   * @param file File
   * @throws IOException I/O exception
   */
  public static void toFile(byte[] b, File file) throws IOException {
    if (!file.exists()) {
      if(!file.createNewFile()) {
        throw new IOException("Can not create file");
      }
    }
    OutputStream os = new FileOutputStream(file);
    exportOutputStream(b, os);
    os.close();
  }

  /**
   * Copy inputstream to outputstream
   *
   * @param is InputStream
   * @param os OutputStream
   * @throws IOException I/O exception
   */
  public static void copy(InputStream is, OutputStream os) throws IOException {
    is.transferTo(os);
    os.flush();
  }
}
