/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import javax.crypto.AEADBadTagException;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.SecretKeySpec;

/**
 * Crypto util
 *
 * https://docs.oracle.com/en/java/javase/11/docs/specs/security/standard-names.html
 *
 * @author Leonard Woo
 */
public class CryptoUtil {

  /**
   * Cipher util
   *
   * @see Cipher
   * @param algorithm Cipher algorithm
   * @param keyAlgorithm Secret key algorithm
   * @param data raw data
   * @param key crypto key
   * @param opmode cipher mode
   * @param params Algorithm Parameter Spec
   * @return data
   * @throws NoSuchPaddingException if transformation contains a padding scheme that is not available.
   * @throws NoSuchAlgorithmException if transformation is null, empty, in an invalid format, or if
   * no Provider supports a CipherSpi implementation for the specified algorithm.
   * @throws InvalidAlgorithmParameterException if the given algorithm parameters are inappropriate
   * for this cipher, or this cipher requires algorithm parameters and params is null, or the given
   * algorithm parameters imply a cryptographic strength that would exceed the legal limits (as
   * determined from the configured jurisdiction policy files).
   * @throws InvalidKeyException if the given key is inappropriate for initializing this cipher, or
   * its keysize exceeds the maximum allowable keysize (as determined from the configured
   * jurisdiction policy files).
   * @throws IllegalBlockSizeException if this cipher is a block cipher, no padding has been
   * requested (only in encryption mode), and the total input length of the data processed by this
   * cipher is not a multiple of block size; or if this encryption algorithm is unable to process
   * the input data provided.
   * @throws BadPaddingException if this cipher is in decryption mode, and (un)padding has been
   * requested, but the decrypted data is not bounded by the appropriate padding bytes.
   * @throws AEADBadTagException – if this cipher is decrypting in an AEAD mode (such as GCM/CCM),
   * and the received authentication tag does not match the calculated value.
   */
  public static byte[] cipher(String algorithm, String keyAlgorithm, byte[] data, byte[] key,
      int opmode, AlgorithmParameterSpec params)
      throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidAlgorithmParameterException,
      InvalidKeyException, IllegalBlockSizeException, BadPaddingException {
    Cipher cipher = Cipher.getInstance(algorithm);
    SecretKeySpec keySpec = new SecretKeySpec(key, keyAlgorithm);
    if (params == null) {
      cipher.init(opmode, keySpec);
    } else {
      cipher.init(opmode, keySpec, params);
    }
    return cipher.doFinal(data);
  }

  /**
   * Cipher util
   *
   * @see Cipher
   * @param algorithm Cipher algorithm
   * @param keyAlgorithm Secret key algorithm
   * @param data raw data
   * @param key crypto key
   * @param opmode cipher mode
   * @return data
   * @throws NoSuchPaddingException if transformation contains a padding scheme that is not available.
   * @throws NoSuchAlgorithmException if transformation is null, empty, in an invalid format, or if
   * no Provider supports a CipherSpi implementation for the specified algorithm.
   * @throws InvalidAlgorithmParameterException if the given algorithm parameters are inappropriate
   * for this cipher, or this cipher requires algorithm parameters and params is null, or the given
   * algorithm parameters imply a cryptographic strength that would exceed the legal limits (as
   * determined from the configured jurisdiction policy files).
   * @throws InvalidKeyException if the given key is inappropriate for initializing this cipher, or
   * its keysize exceeds the maximum allowable keysize (as determined from the configured
   * jurisdiction policy files).
   * @throws IllegalBlockSizeException if this cipher is a block cipher, no padding has been
   * requested (only in encryption mode), and the total input length of the data processed by this
   * cipher is not a multiple of block size; or if this encryption algorithm is unable to process
   * the input data provided.
   * @throws BadPaddingException if this cipher is in decryption mode, and (un)padding has been
   * requested, but the decrypted data is not bounded by the appropriate padding bytes.
   * @throws AEADBadTagException – if this cipher is decrypting in an AEAD mode (such as GCM/CCM),
   * and the received authentication tag does not match the calculated value.
   */
  public static byte[] cipher(String algorithm, String keyAlgorithm, byte[] data, byte[] key,
      int opmode)
      throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidAlgorithmParameterException,
      InvalidKeyException, IllegalBlockSizeException, BadPaddingException {
    return cipher(algorithm, keyAlgorithm, data, key, opmode, null);
  }

  /**
   * Message Digest util
   *
   * @see MessageDigest
   * @param algorithm Message Digest Algorithm
   * @param data raw data
   * @return data hash
   * @throws NoSuchAlgorithmException if no Provider supports a MessageDigestSpi implementation for
   * the specified algorithm.
   */
  public static byte[] md(String algorithm, byte[] data) throws NoSuchAlgorithmException {
    MessageDigest md = MessageDigest.getInstance(algorithm);
    md.update(data);
    return md.digest();
  }

  /**
   * Mac util
   *
   * @see Mac
   * @param data raw data
   * @param salt mac hash salt
   * @param algorithm mac hash algorithm
   * @param keyAlgorithm mac hash key algorithm
   * @return data hash
   * @throws NoSuchAlgorithmException if no Provider supports a MacSpi implementation for the
   * specified algorithm.
   * @throws InvalidKeyException if the given key is inappropriate for initializing this MAC.
   */
  public static byte[] mac(byte[] data, byte[] salt, String algorithm, String keyAlgorithm)
      throws NoSuchAlgorithmException, InvalidKeyException {
    Mac mac = Mac.getInstance(algorithm);
    mac.init(new SecretKeySpec(salt, keyAlgorithm));
    return mac.doFinal(data);
  }

  /**
   * Secret Key Factory util
   *
   * @see SecretKeyFactory
   * @param algorithm Secret Key Factory algorithm
   * @param keySpec KeySpec impl e.g. {@code PBEKeySpec}
   * @return KeySpec data result
   * @throws NoSuchAlgorithmException if no Provider supports a SecretKeyFactorySpi implementation
   * for the specified algorithm.
   * @throws InvalidKeySpecException if the given key specification is inappropriate for this
   * secret-key factory to produce a secret key.
   */
  public static byte[] secretKeyFactory(String algorithm, KeySpec keySpec)
      throws NoSuchAlgorithmException, InvalidKeySpecException {
    return SecretKeyFactory.getInstance(algorithm).generateSecret(keySpec).getEncoded();
  }

  /**
   * Asymmetric cryptography key pair generator
   *
   * @param algorithm Asymmetric cryptography key algorithm
   * @param keysize key size
   * @return key pair
   * @throws NoSuchAlgorithmException if no Provider supports a KeyPairGeneratorSpi implementation
   * for the specified algorithm.
   */
  public static KeyPair keyPairGenerator(String algorithm, int keysize) throws NoSuchAlgorithmException {
    KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(algorithm);
    keyPairGenerator.initialize(keysize);
    return keyPairGenerator.generateKeyPair();
  }

  /**
   * Data Signature
   *
   * @param algorithm signature algorithm
   * @param privateKey signature private key
   * @param rawData data
   * @return signed data
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the specified algorithm
   * @throws InvalidKeyException if the key is invalid
   * @throws SignatureException if this signature object is not initialized properly
   */
  public static byte[] signatureSign(String algorithm, PrivateKey privateKey, byte[] rawData)
      throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
    Signature signature = Signature.getInstance(algorithm);
    signature.initSign(privateKey);
    signature.update(rawData);
    return signature.sign();
  }

  /**
   * Data Signature Verification
   *
   * @param algorithm signature algorithm
   * @param publicKey signature private key
   * @param rawData data
   * @param signedData signed data
   * @return true is verified
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the specified algorithm
   * @throws InvalidKeyException if the key is invalid
   * @throws SignatureException if this signature object is not initialized properly
   */
  public static boolean signatureVerify(String algorithm, PublicKey publicKey, byte[] rawData, byte[] signedData)
      throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
    Signature signature = Signature.getInstance(algorithm);
    signature.initVerify(publicKey);
    signature.update(rawData);
    return signature.verify(signedData);
  }
}
