/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Locale;

/**
 * Datetime format and parse util
 *
 * TemporalAccessor is DateTime super interface.
 * Datetime include {@link java.time.LocalDateTime} {@link java.time.LocalDate}
 * {@link java.time.LocalTime} {@link java.time.OffsetDateTime}
 * {@link java.time.OffsetTime} {@link java.time.ZonedDateTime}
 * {@link java.time.ZoneOffset} {@link java.time.Instant} and more.
 *
 * @author Leonard Woo
 */
public class DatetimeUtil {

  /**
   * Datetime format
   *
   * @param pattern see {@link DateTimeFormatter}
   * @param temporal java.time datetime class
   * @return Datetime formatted string
   */
  public static String format(String pattern, TemporalAccessor temporal) {
    return format(pattern, Locale.getDefault(Locale.Category.FORMAT), temporal);
  }

  /**
   * Datetime format
   *
   * @param pattern see {@link DateTimeFormatter}
   * @param locale datetime local
   * @param temporal java.time datetime class
   * @return Datetime formatted string
   */
  public static String format(String pattern, Locale locale, TemporalAccessor temporal) {
    return DateTimeFormatter.ofPattern(pattern, locale).format(temporal);
  }

  /**
   * Datetime parser
   *
   * @param pattern datetime layout {@link DateTimeFormatter}
   * @param timestamp timestamp
   * @return DateTime interface
   */
  public static TemporalAccessor parse(String pattern, String timestamp){
    return parse(pattern, Locale.getDefault(Locale.Category.FORMAT), timestamp);
  }

  /**
   * Datetime parser
   *
   * @param pattern datetime layout {@link DateTimeFormatter}
   * @param locale timestamp local
   * @param timestamp timestamp
   * @return DateTime interface
   */
  public static TemporalAccessor parse(String pattern, Locale locale, String timestamp){
    return DateTimeFormatter.ofPattern(pattern, locale).parse(timestamp);
  }

  /**
   * Datetime to epoch second
   *
   * @param temporal datetime
   * @return epoch second
   */
  public static long toEpochSecond(TemporalAccessor temporal) {
    return Instant.from(temporal).getEpochSecond();
  }

  /**
   * Datetime to epoch millisecond
   *
   * @param temporal datetime
   * @return epoch millisecond
   */
  public static long toEpochMilliSecond(TemporalAccessor temporal) {
    return Instant.from(temporal).toEpochMilli();
  }

  /**
   * Epoch second parser
   *
   * @param epoch epoch second
   * @param zoneId time zone
   * @return Zoned DateTime
   */
  public static ZonedDateTime parseEpochSecond(long epoch, ZoneId zoneId) {
    return Instant.ofEpochSecond(epoch).atZone(zoneId);
  }

  /**
   * Epoch millisecond parser
   *
   * @param milli epoch millisecond
   * @param zoneId time zone
   * @return Zoned DateTime
   */
  public static ZonedDateTime parseEpochMilliSecond(long milli, ZoneId zoneId) {
    return Instant.ofEpochMilli(milli).atZone(zoneId);
  }

  /**
   * Get UTC now epoch millisecond
   *
   * @return epoch millisecond
   */
  public static long now() {
    return Instant.now().toEpochMilli();
  }

  /**
   * Get now epoch millisecond
   *
   * @param zoneId zone id
   * @return epoch millisecond
   */
  public static long now(ZoneId zoneId) {
    return Instant.now(Clock.system(zoneId)).toEpochMilli();
  }
}
