/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * String Util
 *
 * @author Leonard Woo
 */
public class StringUtil {

  private static final char[] LOWER =
      {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

  private static final char[] UPPER =
      {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

  /**
   * Get a string object with default value
   *
   * @param src string object
   * @param defaultValue default string object
   * @return result
   */
  public static String safeNull(String src, String defaultValue) {
    return ObjectUtil.safeNull(src, defaultValue);
  }

  /**
   * test string is null or length is 0
   *
   * @param str string object
   * @return true is yes
   */
  public static boolean isEmpty (String str) {
    return str == null || "".equals(str);
  }

  /**
   * test string has any char
   *
   * @param str string object
   * @return true is yes
   */
  public static boolean hasLength(String str) {
    return (str != null && !str.isEmpty());
  }

  /**
   * test char sequence has any char without whitespace
   *
   * @param str char sequence object
   * @return true is yes
   */
  public static boolean hasText(CharSequence str) {
    return (str != null && str.length() > 0 && containsText(str));
  }

  /**
   * test string has any char without whitespace
   *
   * @param str string object
   * @return true is yes
   */
  public static boolean hasText(String str) {
    return (str != null && str.length() > 0 && containsText(str));
  }

  /**
   * convert string data from old encoding to new encoding
   *
   * @param data string data
   * @param oldEncoding old encoding
   * @param newEncoding new encoding
   * @return convert result
   */
  public static String transcoding(String data, Charset oldEncoding, Charset newEncoding) {
    return newEncoding.decode( oldEncoding.encode( data ) ).toString();
  }

  private static boolean containsText(CharSequence str) {
    int strLen = str.length();
    for (int i = 0; i < strLen; i++) {
      if (!Character.isWhitespace(str.charAt(i))) {
        return true;
      }
    }
    return false;
  }

  /**
   * return fixed length string object
   *
   * @param str string object
   * @param length length
   * @param preChar pre-padded character
   * @return fixed length string object
   */
  public static String fixedLength(String str, int length, char preChar) {
    StringBuilder sb = new StringBuilder();
    if (str.length() < length) {
      int preLength = length - str.length();
      sb.append(String.valueOf(preChar).repeat(preLength));
      sb.append(str);
    } else {
      sb.append(str.substring(str.length() - length));
    }
    return sb.toString();
  }

  /**
   * capitalize the first letter
   *
   * @param str origin string
   * @return new string
   */
  public static String initialsUpperCase(String str) {
    if (str.length() > 1) {
      return str.substring(0, 1).toUpperCase() + str.substring(1);
    }
    return str.toUpperCase();
  }

  /**
   * replace CharSequence between start and end
   *
   * @param data origin data
   * @param start replace start index
   * @param end replace end index
   * @param replacement replace data
   * @return new data string
   */
  public static String replaceIndex(CharSequence data, int start, int end,
      CharSequence replacement) {
    StringBuilder sb = new StringBuilder();
    sb.append(data, 0, start);
    sb.append(replacement);
    sb.append(data, end, data.length());
    return sb.toString();
  }

    /**
     * convert byte array to hex char array
     *
     * @param data byte array data
     * @return hex char array
     */
  public static char[] encodeHexWithInteger(byte[] data) {
    char[] hexs = new char[data.length * 2];
    for (int i = 0, j = 0; i < data.length; i++, j = j + 2) {
      char[] hex = Integer.toHexString(data[i] & 0xFF).toCharArray();
      hexs[j] = hex.length == 1? '0': hex[0];
      hexs[j + 1] = hex.length == 1? hex[0]: hex[1];
    }
    return hexs;
  }

  /**
   * convert byte array to string object
   *
   * @param data byte array data
   * @return string object
   */
  public static String encodeHexToString(byte[] data) {
    return convertToString(encodeHex(data), 2, " ");
  }

  /**
   * convert byte array to hex char array
   *
   * @param data byte array data
   * @return hex char array
   */
  public static char[] encodeHex(byte[] data) {
    return encodeHex(data, true);
  }

  /**
   * convert byte array to hex char array
   *
   * @param data byte array data
   * @param toLowerCase true is lower case, false is upper case
   * @return hex char array
   */
  public static char[] encodeHex(byte[] data, boolean toLowerCase) {
    return encodeHex(data, toLowerCase? LOWER: UPPER);
  }

  /**
   * Convert char array to String with separate
   *
   * @param src Raw data
   * @param splitNum Separation interval
   * @param split Separator
   * @return encoded string
   */
  public static String convertToString(char[] src, int splitNum, String split) {
    StringBuilder sb = new StringBuilder();
    for (int i = 0; i < src.length; i = i + splitNum) {
      for (int j = 0; j < splitNum; j++) {
        sb.append(src[i + j]);
      }
      sb.append(split);
    }
    String dst = sb.toString();
    return dst.substring(0, dst.length() - split.length());
  }

  /**
   * Delete string Separator and to char array
   *
   * @param src Data
   * @param split Separator
   * @return char array
   */
  public static char[] convertToCharArray(String src, String split) {
    return src.replaceAll(split, "").toCharArray();
  }

  private static char[] encodeHex(byte[] data, char[] hexChar) {
    char[] hex = new char[data.length * 2];
    for (int i = 0, j = 0; i < data.length; i++, j = j + 2) {
      hex[j] = hexChar[(0xF0 & data[i]) >>> 0x04];
      hex[j + 1] = hexChar[0x0F & data[i]];
    }
    return hex;
  }

  /**
   * convert hex char array to byte array
   *
   * @param data hex char array
   * @return byte array
   */
  public static byte[] decodeHex(char[] data) {
    byte[] hex = new byte[data.length / 2];
    for (int i = 0, j = 0; i < hex.length; i++, j = j + 2) {
      int f = toDigit(data[j]) << 0x04;
      f = f | toDigit(data[j + 1]);
      hex[i] = (byte) (f & 0xFF);
    }
    return hex;
  }

  private static int toDigit(char hexChar) {
    int digit = Character.digit(hexChar, 16);
    if(digit == -1) {
      throw new IllegalArgumentException("Invalid Hexadecimal Character: "+ hexChar);
    }
    return digit;
  }

  /**
   * Charset decode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data byte array
   * @return CharBuffer
   * @see Charset
   * @see StandardCharsets
   */
  public static CharBuffer charsetDecode(Charset charset, byte[] data) {
    return charset.decode(ByteBuffer.wrap(data));
  }
}
