/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

/**
 * Object Util
 *
 * @author Leonard Woo
 */
public class ObjectUtil {

  /**
   * return object with default
   *
   * @param t object
   * @param defaultValue default
   * @param <T> object type
   * @return object result
   */
  public static <T> T safeNull(T t, T defaultValue) {
    return isNull(t)? defaultValue: t;
  }

  /**
   * test object is array
   *
   * @param obj object
   * @return true is not null and yes
   */
  public static boolean isArray(Object obj) {
    return notNull(obj) && obj.getClass().isArray();
  }

  /**
   * test object array is empty
   *
   * @param array object array
   * @return true is yes
   */
  public static boolean isEmpty(Object[] array) {
    return (array == null || array.length == 0);
  }

  /**
   * test object is null
   *
   * @param obj object
   * @param <T> object type
   * @return true is yes
   */
  public static <T> boolean isNull(T obj) {
    return obj == null;
  }

  /**
   * test object is not null
   *
   * @param obj object
   * @param <T> object type
   * @return true is yes
   */
  public static <T> boolean notNull(T obj) {
    return obj != null;
  }

  /**
   * Deep copy
   *
   * @param obj source object
   * @return copy object
   * @throws IOException copy failed
   * @throws ClassNotFoundException can not found new object
   */
  public static Object deepCopy(Object obj) throws IOException, ClassNotFoundException {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    ObjectOutputStream oos = new ObjectOutputStream(baos);
    oos.writeObject(obj);
    ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
    ObjectInputStream ois = new ObjectInputStream(bais);
    return ois.readObject();
  }

}
