/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

/**
 * Number Util
 *
 * @author Leonard Woo
 */
public class NumberUtil {

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(byte num, byte min, byte max) {
    return num >= min && num <= max;
  }

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(short num, short min, short max) {
    return num >= min && num <= max;
  }

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(int num, int min, int max) {
    return num >= min && num <= max;
  }

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(long num, long min, long max) {
    return num >= min && num <= max;
  }

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(float num, float min, float max) {
    return num >= min && num <= max;
  }

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(double num, double min, double max) {
    return num >= min && num <= max;
  }

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(BigDecimal num, BigDecimal min, BigDecimal max) {
    return num.compareTo(min) >= 0 && num.compareTo(max) <= 0;
  }

  /**
   * test num between min and max
   *
   * @param num number
   * @param min minimum number
   * @param max maximum number
   * @return true number is between the maximum and minimum
   */
  public static boolean between(BigInteger num, BigInteger min, BigInteger max) {
    return num.compareTo(min) >= 0 && num.compareTo(max) <= 0;
  }

  /**
   * Double format
   *
   * @param precisionPattern precision pattern, e.g. 0.00
   * @param number double number
   * @return formatted string
   */
  public static String format(String precisionPattern, double number) {
    return new DecimalFormat(precisionPattern).format(number);
  }

  /**
   * byte array to unsigned int array
   *
   * @param src byte array
   * @return unsigned int array
   */
  public static int[] toUnsignedIntArray(byte[] src) {
    int[] dst = new int[src.length];
    for (int i = 0; i < src.length; i++) {
      dst[i] = Byte.toUnsignedInt(src[i]);
    }
    return dst;
  }

  /**
   * short array to unsigned int array
   *
   * @param src short array
   * @return unsigned int array
   */
  public static int[] toUnsignedShortArray(short[] src) {
    int[] dst = new int[src.length];
    for (int i = 0; i < src.length; i++) {
      dst[i] = Short.toUnsignedInt(src[i]);
    }
    return dst;
  }

  /**
   * int array to unsigned long array
   *
   * @param src int array
   * @return unsigned long array
   */
  public static long[] toUnsignedLongArray(int[] src) {
    long[] dst = new long[src.length];
    for (int i = 0; i < src.length; i++) {
      dst[i] = Integer.toUnsignedLong(src[i]);
    }
    return dst;
  }

  /**
   * ISO-LATIN-1(ASCII) digits ('0' through '9')
   */
  public static final int LATIN_DIGITS = 1; // 0x30 to 0x39

  /**
   * Fullwidth digits ('０' through '９')
   */
  public static final int FULLWIDTH_DIGITS = 3; // 0xFF10 to 0xFF19

  /**
   * Convert between {@code LATIN_DIGITS} and {@code FULLWIDTH_DIGITS}
   *
   * @param input digit character sequence
   * @param outType output digit type
   * @return digit string
   */
  public static String convert(CharSequence input, int outType) {
    if (notDigit(input)) {
      throw new IllegalArgumentException("input is not number");
    }

    StringBuilder sb = new StringBuilder();
    BigDecimal digit = new BigDecimal( input.toString()
        // Replace first fullwidth numeric symbol
        .replaceFirst("^\\uFF0B", "+")
        .replaceFirst("^\\uFF0D", "-")
        .replaceFirst("\\uFF0E", ".")
    );

    if (outType == LATIN_DIGITS) {
      sb.append(digit);
    } else if (outType == FULLWIDTH_DIGITS) {
      digit.toString().chars().forEach(i -> {
        sb.append((char)(i - 0x30 + 0xFF10));
      });
    } else {
      throw new IllegalArgumentException("outType is wrong");
    }
    return sb.toString();
  }

  private static final char FULLWIDTH_PLUS = 0xFF0B;
  private static final char FULLWIDTH_MINUS = 0xFF0D;
  private static final char FULLWIDTH_FULL_STOP = 0xFF0E;

  private static boolean notDigit(CharSequence input) {
    for (char c : input.toString().toCharArray()) {
      if (!isNumeric(c)) {
        return true;
      }
    }
    return false;
  }

  // is latin or fullwidth numeric
  private static boolean isNumeric(int ch) {
    return between(ch, 0x30, 0x39) ||
        between(ch, 0xFF10, 0xFF19) ||
        isNumericSymbol(ch)
        ;
  }

  // is numeric symbol
  private static boolean isNumericSymbol(int ch) {
    return '.' == ch || FULLWIDTH_FULL_STOP == ch ||
        '-' == ch || FULLWIDTH_MINUS == ch ||
        '+' == ch || FULLWIDTH_PLUS == ch
        ;
  }

}
