/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * String Util
 *
 * @author Leonard Woo
 */
public class StringUtil {

  /**
   * Get a string object with default value
   *
   * @param src string object
   * @param defaultValue default string object
   * @return result
   */
  public static String safeNull(String src, String defaultValue) {
    return ObjectUtil.safeNull(src, defaultValue);
  }

  /**
   * test string is empty or contains only whitespace codepoints
   *
   * @param str string object
   * @return true is yes
   */
  public static boolean isEmpty (CharSequence str) {
    return str == null || str.length() == 0;
  }

  /**
   * test string has any char
   *
   * @param str string object
   * @return true is yes
   */
  public static boolean hasLength(String str) {
    return (str != null && !str.isEmpty());
  }

  /**
   * test char sequence has any char without non-blank character
   *
   * @param str char sequence object
   * @return true is not null and without whitespace
   */
  public static boolean hasText(CharSequence str) {
    return (str != null && str.length() > 0 && containsText(str));
  }

  /**
   * test char sequence is not null and empty or has non-blank character
   *
   * @param str char sequence object
   * @return true is not null and empty
   */
  public static boolean nonText(CharSequence str) {
    return str != null && (str.length() == 0 || !containsText(str));
  }

  private static boolean containsText(CharSequence str) {
    int strLen = str.length();
    for (int i = 0; i < strLen; i++) {
      if (!Character.isWhitespace(str.charAt(i))) {
        return true;
      }
    }
    return false;
  }

  /**
   * Test CharSequence is numeric
   *
   * @param input numeric
   * @return true is numeric
   */
  public static boolean isNumeric(CharSequence input) {
    return Boolean.TRUE.equals(matches("^-?\\d+(\\.\\d+)?$", input));
  }

  /**
   * Test CharSequence is integer
   *
   * @param input numeric
   * @return true is integer
   */
  public static boolean isInteger(CharSequence input) {
    return Boolean.TRUE.equals(matches("^-?\\d+$", input));
  }

  /**
   * Test CharSequence is decimal
   *
   * @param input numeric
   * @return true is decimal
   */
  public static boolean isDecimal(CharSequence input) {
    return Boolean.TRUE.equals(matches("^-?\\d+\\.\\d+$", input));
  }

  /**
   * Test CharSequence is punct
   *
   * @param input punct
   * @return true is punct
   */
  public static boolean isPunct(CharSequence input) {
    return Boolean.TRUE.equals(matches("^\\p{Punct}$", input));
  }

  /**
   * Test CharSequence is symbol
   *
   * @param input symbol
   * @return true is symbol
   */
  public static boolean isSymbol(CharSequence input) {
    return Boolean.TRUE.equals(matches("^\\p{Symbol}$", input));
  }

  /**
   * Compiles the given regular expression and attempts to match the given input against it.
   *
   * @param regex The expression to be compiled
   * @param input The character sequence to be matched
   * @return whether or not the regular expression matches on the input, when the expression's
   *     syntax is invalid is null
   */
  public static Boolean matches(String regex, CharSequence input) {
    try {
      return Pattern.matches(regex, input);
    } catch (PatternSyntaxException ignored) {
    }
    return null;
  }

  /**
   * Convert string data from old encoding to new encoding
   *
   * @param data string data
   * @param oldEncoding old encoding
   * @param newEncoding new encoding
   * @return convert result
   */
  public static String transcoding(String data, Charset oldEncoding, Charset newEncoding) {
    return newEncoding.decode( oldEncoding.encode( data ) ).toString();
  }

  /**
   * Return fixed length string object
   *
   * @param str string object
   * @param length count length
   * @param preChar pre-padded character
   * @return fixed length string object
   */
  public static String fixedLength(String str, int length, char preChar) {
    StringBuilder sb = new StringBuilder();
    if (str.length() < length) {
      int preLength = length - str.length();
      sb.append(String.valueOf(preChar).repeat(preLength));
      sb.append(str);
    } else {
      sb.append(str.substring(str.length() - length));
    }
    return sb.toString();
  }

  /**
   * Capitalize the first letter
   *
   * @param str origin string
   * @return new string
   */
  public static String initialsUpperCase(String str) {
    if (str.length() > 1) {
      return str.substring(0, 1).toUpperCase() + str.substring(1);
    }
    return str.toUpperCase();
  }

  /**
   * Replace CharSequence between start and end
   *
   * @param data origin data
   * @param start replace start index
   * @param end replace end index
   * @param replacement replace data
   * @return new string
   */
  public static String replaceIndex(CharSequence data, int start, int end,
      CharSequence replacement) {
    StringBuilder sb = new StringBuilder();
    sb.append(data, 0, start);
    sb.append(replacement);
    sb.append(data, end, data.length());
    return sb.toString();
  }

  /**
   * Convert char array to String with separate
   *
   * @param src Raw data
   * @param splitNum Separation interval
   * @param split Separator
   * @return encoded string
   */
  public static String convertToString(char[] src, int splitNum, String split) {
    StringBuilder sb = new StringBuilder();
    for (int i = 0; i < src.length; i = i + splitNum) {
      for (int j = 0; j < splitNum; j++) {
        sb.append(src[i + j]);
      }
      sb.append(split);
    }
    String dst = sb.toString();
    return dst.substring(0, dst.length() - split.length());
  }

  /**
   * Delete string Separator and to char array
   *
   * @param src Data
   * @param split Separator
   * @return char array
   */
  public static char[] convertToCharArray(String src, String split) {
    return src.replaceAll(split, "").toCharArray();
  }

  /**
   * Charset decode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data byte array
   * @return character buffer object
   * @see Charset
   * @see StandardCharsets
   */
  public static CharBuffer charsetDecode(Charset charset, byte[] data) {
    try {
      return charset.newDecoder().decode(ByteBuffer.wrap(data));
    } catch (CharacterCodingException ignored) {
    }
    return null;
  }

  /**
   * Charset encode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data character buffer
   * @return byte array
   * @see Charset
   * @see StandardCharsets
   */
  public static byte[] charsetEncode(Charset charset, CharBuffer data) {
    try {
      return charset.newEncoder().encode(data).array();
    } catch (CharacterCodingException ignored) {
    }
    return null;
  }

}
