/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import org.seppiko.commons.utils.crypto.CryptoUtil;
import org.seppiko.commons.utils.crypto.spec.KeySpecUtil;

/**
 * Hash Crypt Util
 *
 * @author Leonard Woo
 */
public class HashCryptUtil {

  /**
   * Simplifies common {@link java.security.MessageDigest} tasks.
   *
   * @param algorithm Message Digest Algorithm
   * @param data data to hash
   * @return complete hash value
   * @throws IllegalArgumentException when a {@link NoSuchAlgorithmException} is caught.
   */
  public static byte[] mdCrypt(MessageDigestAlgorithms algorithm, byte[] data)
      throws IllegalArgumentException {
    try {
      return CryptoUtil.md(algorithm.getName(), data);
    } catch (NoSuchAlgorithmException ignored) {
    }
    throw new IllegalArgumentException();
  }

  /**
   * Simplifies common {@link javax.crypto.Mac} tasks.
   *
   * @param algorithm Hmac Algorithm
   * @param data data to hash with key
   * @param key the keyed digest
   * @return complete hash value including key
   * @throws InvalidKeyException if the given key is {@code null} or does not match the allowed pattern.
   * @throws IllegalArgumentException when a {@link NoSuchAlgorithmException} is caught.
   */
  public static byte[] hmacCrypt(HmacAlgorithms algorithm, byte[] data, byte[] key)
      throws InvalidKeyException, IllegalArgumentException {
    try {
      return CryptoUtil.mac(algorithm.getName(), algorithm.getName(), data, key);
    } catch (NoSuchAlgorithmException ignored) {
    }
    throw new IllegalArgumentException();
  }

  /**
   * Simplifies common PBKDF2 tasks
   *
   * @param algorithm PBKDF2With&lt;Hmac Algorithm&gt;
   * @param password raw password
   * @param salt password salt
   * @param iterations iteration
   * @param keySize key size
   * @return PBKDF2With&lt;Hmac Algorithm&gt; password
   * @throws IllegalArgumentException password or salt is null, number is wrong or could not execute task.
   */
  public static byte[] pbkdf2Crypt(HmacAlgorithms algorithm, char[] password, byte[] salt,
      int iterations, int keySize) {
    try {
      return CryptoUtil.secretKeyFactory("PBKDF2With" + algorithm,
          KeySpecUtil.getPBE(password, salt, iterations, keySize));
    } catch (NoSuchAlgorithmException | InvalidKeySpecException | IllegalArgumentException |
        NullPointerException ex) {
      throw new IllegalArgumentException(ex);
    }
  }

}
