/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import org.seppiko.commons.utils.Environment;
import org.seppiko.commons.utils.StringUtil;

/**
 * Hex Util - Convert byte array to hex char array
 *
 * @author Leonard Woo
 */
public class HexUtil {

  /**
   * convert byte array to hex char array
   *
   * @param data byte array data
   * @return hex char array
   */
  public static char[] encodeWithInteger(byte[] data) {
    char[] hexs = new char[data.length * 2];
    for (int i = 0, j = 0; i < data.length; i++, j = j + 2) {
      char[] hex = Integer.toHexString(data[i] & 0xFF).toCharArray();
      hexs[j] = hex.length == 1? '0': hex[0];
      hexs[j + 1] = hex.length == 1? hex[0]: hex[1];
    }
    return hexs;
  }

  /**
   * convert byte array to string object
   *
   * @param data byte array data
   * @return string object
   */
  public static String encodeToString(byte[] data) {
    return encodeToString(data, " ");
  }

  /**
   * convert byte array to string object
   *
   * @param data byte array data
   * @param split string object split
   * @return string object
   */
  public static String encodeToString(byte[] data, String split) {
    return StringUtil.convertToString(encode(data), 2, split);
  }

  /**
   * convert byte array to hex char array
   *
   * @param data byte array data
   * @return hex char array
   */
  public static char[] encode(byte[] data) {
    return encode(data, true);
  }

  /**
   * convert byte array to hex char array
   *
   * @param data byte array data
   * @param toLowerCase true is lower case, false is upper case
   * @return hex char array
   */
  public static char[] encode(byte[] data, boolean toLowerCase) {
    return encode(data, toLowerCase? Environment.HEXADECIMAL: Environment.HEXADECIMAL_UPPER);
  }

  private static char[] encode(byte[] data, char[] hexChar) {
    char[] hex = new char[data.length * 2];
    for (int i = 0, j = 0; i < data.length; i++, j = j + 2) {
      hex[j] = hexChar[(0xF0 & data[i]) >>> 0x04];
      hex[j + 1] = hexChar[0x0F & data[i]];
    }
    return hex;
  }

  /**
   * convert hex string with split to byte array
   *
   * @param data hex string
   * @param split split
   * @return byte array
   */
  public static byte[] decodeString(String data, String split) {
    return decode(data.replaceAll(split, "").toCharArray());
  }

  /**
   * convert hex char array to byte array
   *
   * @param data hex char array
   * @return byte array
   */
  public static byte[] decode(char[] data) {
    byte[] hex = new byte[data.length / 2];
    for (int i = 0, j = 0; i < hex.length; i++, j = j + 2) {
      int f = toDigit(data[j]) << 0x04;
      f = f | toDigit(data[j + 1]);
      hex[i] = (byte) (f & 0xFF);
    }
    return hex;
  }

  private static int toDigit(char hexChar) {
    int digit = Character.digit(hexChar, 16);
    if(digit == -1) {
      throw new IllegalArgumentException("Invalid Hexadecimal Character: "+ hexChar);
    }
    return digit;
  }
}
