/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.crypto;

import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.SecureRandomParameters;
import java.security.Security;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import javax.crypto.AEADBadTagException;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKeyFactory;
import org.seppiko.commons.utils.ObjectUtil;
import org.seppiko.commons.utils.StringUtil;
import org.seppiko.commons.utils.crypto.spec.KeySpecUtil;

/**
 * Crypto util
 *
 * @see <a href="https://docs.oracle.com/en/java/javase/11/docs/specs/security/standard-names.html">Java Security Standard Algorithm Names</a>
 * @author Leonard Woo
 */
public class CryptoUtil {

  /**
   * non provider
   */
  public static final Provider NONPROVIDER = null;

  /**
   * Check and get provider object
   *
   * @param providerName the name of the provider
   * @return the provider
   * @throws IllegalArgumentException if provider name is null or empty.
   * @throws NoSuchProviderException if it can not load provider.
   */
  public static Provider getProvider(String providerName)
      throws IllegalArgumentException, NoSuchProviderException {
    if (StringUtil.isEmpty(providerName)) {
      throw new IllegalArgumentException();
    }
    Provider p = Security.getProvider(providerName);
    if (p == null) {
      throw new NoSuchProviderException();
    }
    return p;
  }

  /**
   * Cipher util
   *
   * @see Cipher
   * @param algorithm cipher algorithm
   * @param provider cipher provider, if you do not need this set {@code null} or {@code CryptoUtil.NONPROVIDER}
   *        see {@code CryptoUtil.getProvider(providerName)}
   * @param opmode cipher mode
   * @param key crypto key
   * @param params Algorithm Parameter Spec, if you do not need this set null
   * @param data raw data
   * @return data
   * @throws NoSuchPaddingException if transformation contains a padding scheme that is not available.
   * @throws NoSuchAlgorithmException if transformation is null, empty, in an invalid format, or if
   * no Provider supports a CipherSpi implementation for the specified algorithm.
   * @throws InvalidAlgorithmParameterException if the given algorithm parameters are inappropriate
   * for this cipher, or this cipher requires algorithm parameters and params is null, or the given
   * algorithm parameters imply a cryptographic strength that would exceed the legal limits (as
   * determined from the configured jurisdiction policy files).
   * @throws InvalidKeyException if the given key is inappropriate for initializing this cipher, or
   * its keysize exceeds the maximum allowable keysize (as determined from the configured
   * jurisdiction policy files).
   * @throws IllegalBlockSizeException if this cipher is a block cipher, no padding has been
   * requested (only in encryption mode), and the total input length of the data processed by this
   * cipher is not a multiple of block size; or if this encryption algorithm is unable to process
   * the input data provided.
   * @throws BadPaddingException if this cipher is in decryption mode, and (un)padding has been
   * requested, but the decrypted data is not bounded by the appropriate padding bytes.
   * @throws AEADBadTagException – if this cipher is decrypting in an AEAD mode (such as GCM/CCM),
   * and the received authentication tag does not match the calculated value.
   */
  public static byte[] cipher(String algorithm, Provider provider, int opmode, Key key,
      AlgorithmParameterSpec params, byte[] data)
      throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException,
      InvalidAlgorithmParameterException, IllegalBlockSizeException, BadPaddingException {

    Cipher cipher = ObjectUtil.isNull(provider)?
        Cipher.getInstance(algorithm):
        Cipher.getInstance(algorithm, provider);

    if (ObjectUtil.isNull(params)) {
      cipher.init(opmode, key);
    } else {
      cipher.init(opmode, key, params);
    }

    return cipher.doFinal(data);
  }

  /**
   * Cipher util
   *
   * @see Cipher
   * @param algorithm cipher algorithm
   * @param providerName cipher provider name, if you do not need this set {@code null} or {@code ""}
   * @param opmode cipher mode
   * @param key crypto key
   * @param params Algorithm Parameter Spec, if you do not need this set null
   * @param data raw data
   * @return data
   * @throws NoSuchPaddingException if transformation contains a padding scheme that is not
   *     available.
   * @throws NoSuchAlgorithmException if transformation is null, empty, in an invalid format, or if
   *     no Provider supports a CipherSpi implementation for the specified algorithm.
   * @throws InvalidAlgorithmParameterException if the given algorithm parameters are inappropriate
   *     for this cipher, or this cipher requires algorithm parameters and params is null, or the
   *     given algorithm parameters imply a cryptographic strength that would exceed the legal
   *     limits (as determined from the configured jurisdiction policy files).
   * @throws InvalidKeyException if the given key is inappropriate for initializing this cipher, or
   *     its keysize exceeds the maximum allowable keysize (as determined from the configured
   *     jurisdiction policy files).
   * @throws IllegalBlockSizeException if this cipher is a block cipher, no padding has been
   *     requested (only in encryption mode), and the total input length of the data processed by
   *     this cipher is not a multiple of block size; or if this encryption algorithm is unable to
   *     process the input data provided.
   * @throws BadPaddingException if this cipher is in decryption mode, and (un)padding has been
   *     requested, but the decrypted data is not bounded by the appropriate padding bytes.
   * @throws AEADBadTagException – if this cipher is decrypting in an AEAD mode (such as GCM/CCM),
   *     and the received authentication tag does not match the calculated value.
   */
  public static byte[] cipher(String algorithm, String providerName, int opmode, Key key,
      AlgorithmParameterSpec params, byte[] data)
      throws InvalidAlgorithmParameterException, NoSuchPaddingException, IllegalBlockSizeException,
          NoSuchAlgorithmException, BadPaddingException, InvalidKeyException {

    Provider p = StringUtil.isEmpty(providerName)? NONPROVIDER: Security.getProvider(providerName);

    return cipher(algorithm, p, opmode, key, params, data);
  }

  /**
   * Message Digest util
   *
   * @see MessageDigest
   * @param algorithm Message Digest Algorithm
   * @param data raw data
   * @return data hash
   * @throws NoSuchAlgorithmException if no Provider supports a MessageDigestSpi implementation for
   *     the specified algorithm.
   * @throws NullPointerException if algorithm is null.
   */
  public static byte[] md(String algorithm, byte[] data) throws NoSuchAlgorithmException,
      NullPointerException {
    MessageDigest md = MessageDigest.getInstance(algorithm);
    md.update(data);
    return md.digest();
  }

  /**
   * Message Digest util
   *
   * @see MessageDigest
   * @param algorithm Message Digest Algorithm
   * @param provider the provider
   * @param data raw data
   * @return data hash
   * @throws NoSuchAlgorithmException if no Provider supports a MessageDigestSpi implementation for
   *     the specified algorithm.
   * @throws NullPointerException if algorithm is null.
   * @throws IllegalArgumentException if provider is null.
   */
  public static byte[] md(String algorithm, Provider provider, byte[] data)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException {
    MessageDigest md = MessageDigest.getInstance(algorithm, provider);
    md.update(data);
    return md.digest();
  }

  /**
   * Mac util
   *
   * @see Mac
   * @param algorithm mac hash algorithm
   * @param keyAlgorithm mac hash key algorithm
   * @param data raw data
   * @param key mac hash key
   * @return data hash
   * @throws NoSuchAlgorithmException if no Provider supports a MacSpi implementation for the
   *     specified algorithm.
   * @throws InvalidKeyException if the given key is inappropriate for initializing this MAC.
   * @throws NullPointerException if algorithm is null.
   */
  public static byte[] mac(String algorithm, String keyAlgorithm, byte[] data, byte[] key)
      throws NoSuchAlgorithmException, InvalidKeyException, NullPointerException {
    Mac mac = Mac.getInstance(algorithm);
    mac.init(KeySpecUtil.getSecret(key, keyAlgorithm));
    return mac.doFinal(data);
  }

  /**
   * Mac util
   *
   * @see Mac
   * @param algorithm mac hash algorithm
   * @param provider the provider
   * @param keyAlgorithm mac hash key algorithm
   * @param data raw data
   * @param key mac hash key
   * @return data hash
   * @throws NoSuchAlgorithmException if no Provider supports a MacSpi implementation for the
   *     specified algorithm.
   * @throws InvalidKeyException if the given key is inappropriate for initializing this MAC.
   * @throws NullPointerException if algorithm is null.
   */
  public static byte[] mac(String algorithm, Provider provider, String keyAlgorithm, byte[] data,
      byte[] key)
      throws NoSuchAlgorithmException, InvalidKeyException, NullPointerException {
    Mac mac = Mac.getInstance(algorithm, provider);
    mac.init(KeySpecUtil.getSecret(key, keyAlgorithm));
    return mac.doFinal(data);
  }

  /**
   * Secret Key Factory util
   *
   * @see SecretKeyFactory
   * @param algorithm Secret Key Factory algorithm
   * @param keySpec KeySpec impl e.g. {@code PBEKeySpec}
   * @return KeySpec data result
   * @throws NoSuchAlgorithmException if no Provider supports a SecretKeyFactorySpi implementation
   *     for the specified algorithm.
   * @throws InvalidKeySpecException if the given key specification is inappropriate for this
   *     secret-key factory to produce a secret key.
   * @throws NullPointerException if algorithm is null.
   */
  public static byte[] secretKeyFactory(String algorithm, KeySpec keySpec)
      throws NoSuchAlgorithmException, InvalidKeySpecException, NullPointerException {
    return SecretKeyFactory.getInstance(algorithm).generateSecret(keySpec).getEncoded();
  }

  /**
   * Secret Key Factory util
   *
   * @see SecretKeyFactory
   * @param algorithm Secret Key Factory algorithm
   * @param provider the provider
   * @param keySpec KeySpec impl e.g. {@code PBEKeySpec}
   * @return KeySpec data result
   * @throws NoSuchAlgorithmException if no Provider supports a SecretKeyFactorySpi implementation
   *     for the specified algorithm.
   * @throws InvalidKeySpecException if the given key specification is inappropriate for this
   *     secret-key factory to produce a secret key.
   * @throws IllegalArgumentException if the provider is null.
   * @throws NullPointerException if algorithm is null.
   */
  public static byte[] secretKeyFactory(String algorithm, Provider provider, KeySpec keySpec)
      throws NoSuchAlgorithmException, InvalidKeySpecException, IllegalArgumentException,
      NullPointerException {
    return SecretKeyFactory.getInstance(algorithm, provider).generateSecret(keySpec).getEncoded();
  }

  /**
   * Returns a KeyFactory object that converts public/private keys of the specified algorithm.
   *
   * @param algorithm key algorithm
   * @return KeyFactory object
   * @throws NullPointerException if algorithm is null.
   * @throws NoSuchAlgorithmException the name of the requested key algorithm.
   */
  public static KeyFactory keyFactory(String algorithm)
      throws NullPointerException, NoSuchAlgorithmException {
    return KeyFactory.getInstance(algorithm);
  }

  /**
   * Returns a KeyFactory object that converts public/private keys of the specified algorithm.
   *
   * @param algorithm key algorithm
   * @param provider algorithm provider
   * @return KeyFactory object
   * @throws NullPointerException if algorithm is null
   * @throws NoSuchAlgorithmException if a KeyFactorySpi implementation for the specified algorithm
   *     is not available from the specified provider
   * @throws IllegalArgumentException if the specified provider is null
   */
  public static KeyFactory keyFactory(String algorithm, Provider provider)
      throws NullPointerException, NoSuchAlgorithmException, IllegalArgumentException {
    return KeyFactory.getInstance(algorithm, provider);
  }

  /**
   * Cryptographic key generator
   *
   * @param algorithm the standard name of the requested key algorithm
   * @param keysize key size
   * @return secret key byte array
   * @throws NoSuchAlgorithmException if a KeyGeneratorSpi implementation for the specified
   *     algorithm is not available from the specified Provider object
   * @throws NullPointerException if algorithm is null
   */
  public static byte[] keyGenerator(String algorithm, int keysize)
      throws NoSuchAlgorithmException, NullPointerException {
    KeyGenerator keyGenerator = GeneratorUtil.keyGenerator(algorithm);
    keyGenerator.init(keysize);
    return keyGenerator.generateKey().getEncoded();
  }

  /**
   * Cryptographic key generator
   *
   * @param algorithm the standard name of the requested key algorithm
   * @param provider the provider
   * @param keysize key size
   * @return secret key byte array
   * @throws NoSuchAlgorithmException if a KeyGeneratorSpi implementation for the specified
   *     algorithm is not available from the specified Provider object
   * @throws NullPointerException if algorithm is null
   * @throws IllegalArgumentException if the provider is null
   */
  public static byte[] keyGenerator(String algorithm, Provider provider, int keysize)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException {
    KeyGenerator keyGenerator = GeneratorUtil.keyGenerator(algorithm, provider);
    keyGenerator.init(keysize);
    return keyGenerator.generateKey().getEncoded();
  }

  /**
   * Asymmetric cryptographic key pair generator
   *
   * @see KeyPairGenerator
   * @param algorithm the standard string name of the algorithm
   * @param keysize key size
   * @return key pair
   * @throws NoSuchAlgorithmException if no Provider supports a KeyPairGeneratorSpi implementation
   *     for the specified algorithm.
   */
  public static KeyPair keyPairGenerator(String algorithm, int keysize)
      throws NoSuchAlgorithmException {
    KeyPairGenerator keyPairGenerator = GeneratorUtil.keyPairGenerator(algorithm);
    keyPairGenerator.initialize(keysize);
    return keyPairGenerator.generateKeyPair();
  }

  /**
   * Asymmetric cryptographic key pair generator
   *
   * @see KeyPairGenerator
   * @param algorithm the standard string name of the algorithm
   * @param provider the provider
   * @param keysize key size
   * @return key pair
   * @throws NoSuchAlgorithmException if no Provider supports a KeyPairGeneratorSpi implementation
   *     for the specified algorithm.
   * @throws IllegalArgumentException if the provider is null.
   * @throws NullPointerException if algorithm is null.
   */
  public static KeyPair keyPairGenerator(String algorithm, Provider provider, int keysize)
      throws NoSuchAlgorithmException, IllegalArgumentException, NullPointerException {
    KeyPairGenerator keyPairGenerator = GeneratorUtil.keyPairGenerator(algorithm, provider);
    keyPairGenerator.initialize(keysize);
    return keyPairGenerator.generateKeyPair();
  }

  /**
   *
   * @param algorithm the name of the RNG algorithm.
   * @param seedSize the number of seed bytes to generate.
   * @return the seed bytes.
   * @throws NoSuchAlgorithmException if no Provider supports the specified algorithm.
   * @throws NullPointerException if algorithm is null or empty.
   * @throws IllegalArgumentException if seedSize is negative.
   */
  public static byte[] secureRandom(String algorithm, int seedSize)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException {
    if (StringUtil.isEmpty(algorithm)) {
      throw new NullPointerException();
    }
    return GeneratorUtil.secureRandom(algorithm).generateSeed(seedSize);
  }

  /**
   *
   * @param algorithm the name of the RNG algorithm.
   * @param provider the provider.
   * @param seedSize the number of seed bytes to generate.
   * @return the seed bytes.
   * @throws NoSuchAlgorithmException if no Provider supports the specified algorithm.
   * @throws NullPointerException if algorithm is null or empty, if the specified provider is null.
   * @throws IllegalArgumentException if seedSize is negative.
   */
  public static byte[] secureRandom(String algorithm, Provider provider, int seedSize)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException {
    if (StringUtil.isEmpty(algorithm) || provider == null) {
      throw new NullPointerException();
    }
    return GeneratorUtil.secureRandom(algorithm, provider).generateSeed(seedSize);
  }

  /**
   *
   * @param algorithm the name of the RNG algorithm.
   * @param provider the provider.
   * @param params the SecureRandomParameters the newly created SecureRandom object must support.
   * @param seedSize the number of seed bytes to generate.
   * @return the seed bytes.
   * @throws NoSuchAlgorithmException if no Provider supports the specified algorithm.
   * @throws NullPointerException if algorithm is null or empty, if the specified provider or
   *     params is null.
   * @throws IllegalArgumentException if seedSize is negative.
   */
  public static byte[] secureRandom(String algorithm, Provider provider,
      SecureRandomParameters params, int seedSize)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException {
    if (StringUtil.isEmpty(algorithm) || provider == null || params == null) {
      throw new NullPointerException();
    }
    return GeneratorUtil.secureRandom(algorithm, provider, params).generateSeed(seedSize);
  }

}
