/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.crypto;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;

/**
 * Signature Util
 *
 * @see <a href="https://docs.oracle.com/en/java/javase/11/docs/specs/security/standard-names.html">Java Security Standard Algorithm Names</a>
 * @author Leonard Woo
 */
public class SignatureUtil {

  /**
   * Returns a Signature object that implements the specified signature algorithm.
   *
   * @param algorithm the standard name of the algorithm requested
   * @return the new Signature object
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the
   *     specified algorithm
   * @throws NullPointerException if algorithm is null.
   */
  public static Signature signature(String algorithm)
      throws NoSuchAlgorithmException, NullPointerException {
    return Signature.getInstance(algorithm);
  }

  /**
   * Returns a Signature object that implements the specified signature algorithm.
   *
   * @param algorithm the standard name of the algorithm requested
   * @param provider the provider
   * @return the new Signature object
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the
   *     specified algorithm
   * @throws NullPointerException if algorithm is null.
   * @throws IllegalArgumentException if the provider is null.
   */
  private static Signature signature(String algorithm, Provider provider)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException {
    return Signature.getInstance(algorithm, provider);
  }

  /**
   * Data Signature Encrypt
   *
   * @see Signature
   * @param algorithm signature algorithm
   * @param privateKey signature private key
   * @param rawData data
   * @return signed data
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the
   *     specified algorithm.
   * @throws InvalidKeyException if the key is invalid.
   * @throws SignatureException if this signature object is not initialized properly.
   * @throws NullPointerException if algorithm is null.
   */
  public static byte[] signatureSign(String algorithm, PrivateKey privateKey, byte[] rawData)
      throws NoSuchAlgorithmException, NullPointerException, InvalidKeyException, SignatureException {
    Signature signature = signature(algorithm);
    signature.initSign(privateKey);
    signature.update(rawData);
    return signature.sign();
  }

  /**
   * Data Signature Verification
   *
   * @see Signature
   * @param algorithm signature algorithm
   * @param publicKey signature private key
   * @param rawData data
   * @param signedData signed data
   * @return true is verified
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the
   *     specified algorithm.
   * @throws InvalidKeyException if the key is invalid.
   * @throws SignatureException if this signature object is not initialized properly.
   * @throws NullPointerException if algorithm is null.
   */
  public static boolean signatureVerify(String algorithm, PublicKey publicKey, byte[] rawData,
      byte[] signedData)
      throws NoSuchAlgorithmException, NullPointerException, InvalidKeyException, SignatureException {
    Signature signature = signature(algorithm);
    signature.initVerify(publicKey);
    signature.update(rawData);
    return signature.verify(signedData);
  }

  /**
   * Data Signature Encrypt
   *
   * @see Signature
   * @param algorithm signature algorithm
   * @param provider the provider
   * @param privateKey signature private key
   * @param rawData data
   * @return signed data
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the
   *     specified algorithm.
   * @throws InvalidKeyException if the key is invalid.
   * @throws SignatureException if this signature object is not initialized properly.
   * @throws NullPointerException if algorithm is null.
   */
  public static byte[] signatureSign(String algorithm, Provider provider, PrivateKey privateKey,
      byte[] rawData)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException,
      InvalidKeyException, SignatureException {
    Signature signature = signature(algorithm, provider);
    signature.initSign(privateKey);
    signature.update(rawData);
    return signature.sign();
  }

  /**
   * Data Signature Verification
   *
   * @see Signature
   * @param algorithm signature algorithm
   * @param provider the provider
   * @param publicKey signature private key
   * @param rawData data
   * @param signedData signed data
   * @return true is verified
   * @throws NoSuchAlgorithmException if no Provider supports a Signature implementation for the
   *     specified algorithm.
   * @throws InvalidKeyException if the key is invalid.
   * @throws SignatureException if this signature object is not initialized properly.
   * @throws NullPointerException if algorithm is null.
   * @throws IllegalArgumentException if the provider is null.
   */
  public static boolean signatureVerify(String algorithm, Provider provider, PublicKey publicKey,
      byte[] rawData,  byte[] signedData)
      throws NoSuchAlgorithmException, NullPointerException, IllegalArgumentException,
      InvalidKeyException, SignatureException {
    Signature signature = signature(algorithm, provider);
    signature.initVerify(publicKey);
    signature.update(rawData);
    return signature.verify(signedData);
  }

}
