/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpClient.Version;
import java.net.http.HttpHeaders;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import javax.net.ssl.SSLSession;
import org.seppiko.commons.utils.Assert;
import org.seppiko.commons.utils.ObjectUtil;

/**
 * Http Async Response Util with java.net.http
 *
 * HttpRequest see {@link HttpClientUtil}
 *
 * @author Leonard Woo
 */
public class AsyncHttpClientUtil {

  /**
   * Get Async String Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpClientException get async response exception
   */
  public static HttpResponse<String> getAsyncResponseString(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    CompletableFuture<HttpResponse<String>> resp = (ObjectUtil.isNull(proxy)?
            getAsyncResponseRaw(req, BodyHandlers.ofString()):
            getAsyncResponseProxyRaw(req, BodyHandlers.ofString(), proxy) );
    try {
      return httpResponse(resp);
    } catch (HttpResponseException ex) {
      throw new HttpClientException("Response exception", ex);
    }
  }

  /**
   * Get Async byte array Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpClientException get async response exception
   */
  public static HttpResponse<byte[]> getAsyncResponseByteArray(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    CompletableFuture<HttpResponse<byte[]>> resp = (ObjectUtil.isNull(proxy)?
            getAsyncResponseRaw(req, BodyHandlers.ofByteArray()):
            getAsyncResponseProxyRaw(req, BodyHandlers.ofByteArray(), proxy) );
    try {
      return httpResponse(resp);
    } catch (HttpResponseException ex) {
      throw new HttpClientException("Response exception", ex);
    }
  }

  /**
   * Get Async InputStream Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpClientException get async response exception
   */
  public static HttpResponse<InputStream> getAsyncResponseInputStream(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    CompletableFuture<HttpResponse<InputStream>> resp = (ObjectUtil.isNull(proxy)?
            getAsyncResponseRaw(req, BodyHandlers.ofInputStream()):
            getAsyncResponseProxyRaw(req, BodyHandlers.ofInputStream(), proxy) );
    try {
      return httpResponse(resp);
    } catch (HttpResponseException ex) {
      throw new HttpClientException("Response exception", ex);
    }
  }

  private static <T> HttpResponse<T> httpResponse(CompletableFuture<HttpResponse<T>> resp)
      throws HttpResponseException {

    try {
      return new HttpResponse<>() {

        @Override
        public int statusCode() {
          try {
            return resp.thenApply(HttpResponse::statusCode).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }

        @Override
        public HttpRequest request() {
          try {
            return resp.thenApply(HttpResponse::request).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }

        @Override
        public Optional<HttpResponse<T>> previousResponse() {
          try {
            return resp.thenApply(HttpResponse::previousResponse).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }

        @Override
        public HttpHeaders headers() {
          try {
            return resp.thenApply(HttpResponse::headers).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }

        @Override
        public T body() {
          try {
            return resp.thenApply(HttpResponse::body).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }

        @Override
        public Optional<SSLSession> sslSession() {
          try {
            return resp.thenApply(HttpResponse::sslSession).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }

        @Override
        public URI uri() {
          try {
            return resp.thenApply(HttpResponse::uri).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }

        @Override
        public Version version() {
          try {
            return resp.thenApply(HttpResponse::version).get();
          } catch (InterruptedException | ExecutionException e) {
            throw Assert.sneakyThrow(e);
          }
        }
      };
    } catch (RuntimeException ex) {
      throw new HttpResponseException(ex);
    }
  }

  private static <T> CompletableFuture<HttpResponse<T>> getAsyncResponseRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler) {
    return HttpClient.newHttpClient().sendAsync(req, responseBodyHandler);
  }

  private static <T> CompletableFuture<HttpResponse<T>> getAsyncResponseProxyRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler, InetSocketAddress proxy) {
    return HttpClient.newBuilder().proxy(ProxySelector.of(proxy)).build().sendAsync(req, responseBodyHandler);
  }

}
