/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

/**
 * Simple Http Headers implement for {@link java.net.http.HttpRequest}
 *
 * @author Leonard Woo
 */
public class HttpHeaders {

  /**
   * Empty http headers
   */
  public static final HttpHeaders NO_HEADERS = new HttpHeaders();

  private final LinkedHashMap<String, ArrayList<String>> headers = new LinkedHashMap<>();
  private final String[] EMPTY_STRING_ARRAY = new String[]{};

  /**
   * HttpHeaders
   */
  private HttpHeaders() {
  }

  /**
   * Create HttpHeader object
   *
   * @param name header name
   * @param values header values
   * @return HttpHeaders object
   */
  public static HttpHeaders setHeader(String name, String... values) {
    HttpHeaders headers = new HttpHeaders();
    headers.addHeader(name, values);
    return headers;
  }

  /**
   * Set header
   *
   * @param name header name
   * @param values header values
   */
  public void setHeaders(String name, String... values) {
    headers.put(name, new ArrayList<>(List.of(values)));
  }

  /**
   * Add headers
   *
   * @param name header name
   * @param values header values
   */
  public void addHeader(String name, String... values) {
    if (headers.containsKey(name)) {
      ArrayList<String> valueOld = headers.get(name);
      ArrayList<String> valueNew = (ArrayList<String>) List.of(values);
      for (String s : valueOld) {
        if (!valueNew.contains(s)) {
          valueNew.add(s);
        }
      }
      headers.put(name, valueNew);
    } else {
      setHeaders(name, values);
    }
  }

  /**
   * Get {@link java.net.http.HttpRequest} headers
   *
   * @return Headers
   */
  public String[] getHeaderList() {
    return (headers.isEmpty())? EMPTY_STRING_ARRAY: headerHeadler(headers);
  }

  private String[] headerHeadler(LinkedHashMap<String, ArrayList<String>> headerMap) {
    ArrayList<String> headerList = new ArrayList<>();
    headerMap.forEach((key, value) -> {
      headerList.add(key);
      headerList.add(valuesHeadler(value));
    });

    return headerList.toArray(String[]::new);
  }

  private static final char HTTP_HEADER_VALUE_SEPARTOR = ';';
  private String valuesHeadler(ArrayList<String> valueList) {
    StringBuilder sb = new StringBuilder();
    valueList.forEach(e -> {
      sb.append(e);
      sb.append(HTTP_HEADER_VALUE_SEPARTOR);
      sb.append(" ");
    });

    return sb.deleteCharAt( sb.lastIndexOf(HTTP_HEADER_VALUE_SEPARTOR + "") )
        .toString().stripTrailing();
  }
}
