/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.nio.ByteBuffer;

/**
 * Bytes Util
 *
 * @author Leonard Woo
 */
public class BytesUtil {

  private static final ByteBuffer buffer = ByteBuffer.allocate(Long.BYTES);

 /**
  * Convert long to byte array
  *
  * @param l long number
  * @return byte array
  */
  public static byte[] longToBytes(long l) {
    buffer.putLong(0, l);
    return buffer.array();
  }

 /**
  * Convert byte array to long
  *
  * @param bs byte array
  * @return long number
  */
  public static long bytesToLong(byte[] bs) {
    buffer.put(bs, 0, bs.length);
    buffer.flip();
    return buffer.getLong();
  }

  /**
   * Convert int to byte array
   *
   * @param i int number
   * @return byte array
   */
  public static byte[] intToBytes(int i) {
    buffer.putInt(0, i);
    return buffer.array();
  }

  /**
   * Convert byte array to int
   *
   * @param bs byte array
   * @return int number
   */
  public static int bytesToInt(byte[] bs) {
    buffer.put(bs, 0, bs.length);
    buffer.flip();
    return buffer.getInt();
  }

  /**
   * Convert short to byte array
   *
   * @param s short number
   * @return byte array
   */
  public static byte[] shortToBytes(short s) {
    buffer.putShort(0, s);
    return buffer.array();
  }

  /**
   * Convert byte array to short
   *
   * @param bs byte array
   * @return short number
   */
  public static short bytesToShort(byte[] bs) {
    buffer.put(bs, 0, bs.length);
    buffer.flip();
    return buffer.getShort();
  }

  /**
   * Convert double to byte array
   *
   * @param d double number
   * @return byte array
   */
  public static byte[] doubleToBytes(double d) {
    buffer.putDouble(0, d);
    return buffer.array();
  }

  /**
   * Convert byte array to double
   *
   * @param bs byte array
   * @return double number
   */
  public static double bytesToDouble(byte[] bs) {
    buffer.put(bs, 0, bs.length);
    buffer.flip();
    return buffer.getDouble();
  }

  /**
   * Convert float to byte array
   *
   * @param f float number
   * @return byte array
   */
  public static byte[] floatToBytes(float f) {
    buffer.putFloat(0, f);
    return buffer.array();
  }

  /**
   * Convert byte array to float
   *
   * @param bs byte array
   * @return float number
   */
  public static float bytesToFloat(byte[] bs) {
    buffer.put(bs, 0, bs.length);
    buffer.flip();
    return buffer.getFloat();
  }

}
