/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

/**
 * Base62 util
 *
 * @author Leonard Woo
 */
public class Base62 {

  private static final int BASE_COUNT = 62;

  private static final char[] ALPHABET_BASE = new char[] {
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
      'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
      'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'
  };

  /**
   * encode number to Base62 string
   *
   * @param source number
   * @return Base62
   */
  public static String encodeToString(long source) {
    StringBuilder sb = new StringBuilder();
    if (source == 0) {
      sb.append('a');
    }
    while (source > 0) {
      long mod = source % BASE_COUNT;
      sb.append(ALPHABET_BASE[(int) mod]);
      source = source / BASE_COUNT;
    }
    return sb.toString();
  }

  /**
   * decode Base62 string to number
   *
   * @param str Base62
   * @return number
   */
  public static long decodeFromString(String str) {
    return decode(str.toCharArray());
  }

  private static long decode(char[] chars) {
    long n = 0;
    int pow = 0;
    for(char item: chars) {
      int i = 0;
      if(item >= 'a' && item <= 'z') {
        i = item - 'a';
      }
      if(item >= '0' && item <= '9') {
        i = item - '0' + 26;
      }
      if(item >= 'A' && item <= 'Z') {
        i = item - 'A' + 36;
      }
      n += i * (long) Math.pow(BASE_COUNT, pow);
      pow++;
    }
    return n;
  }

}
