/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.time.Duration;
import org.seppiko.commons.utils.ObjectUtil;

/**
 * Http Client Util with java.net.http
 *
 * @author Leonard Woo
 */
public class HttpClientUtil {

  /**
   * Get Request object with String
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body string
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   * @throws IllegalArgumentException something not supported or undefined
   * @throws NullPointerException something is null
   * @throws HttpRuntimeException Http request process exception
   */
  public static HttpRequest getRequest(String url, HttpMethod method, int timeout,
      HttpHeaders headers, String requestBody)
      throws URISyntaxException, IllegalArgumentException, NullPointerException,
      HttpRuntimeException {

    if (ObjectUtil.isNull(headers)) {
      headers = HttpHeaders.NO_HEADERS;
    }

    HttpRequest.BodyPublisher requestBodyPublisher = ObjectUtil.notNull(requestBody)
            ? HttpRequest.BodyPublishers.ofString(requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(new URI(url), method, timeout, headers.getHeaderList(),
        requestBodyPublisher);
  }

  /**
   * Get Request object with inputstream
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body inputstream
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   * @throws IllegalArgumentException something not supported or undefined
   * @throws NullPointerException something is null
   * @throws HttpRuntimeException Http request process exception
   */
  public static HttpRequest getRequest(
      String url, HttpMethod method, int timeout, HttpHeaders headers, InputStream requestBody)
      throws URISyntaxException, IllegalArgumentException, NullPointerException,
          HttpRuntimeException {

    if (ObjectUtil.isNull(headers)) {
      headers = HttpHeaders.NO_HEADERS;
    }

    HttpRequest.BodyPublisher requestBodyPublisher = ObjectUtil.notNull(requestBody)
            ? HttpRequest.BodyPublishers.ofInputStream(() -> requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(new URI(url), method, timeout, headers.getHeaderList(), requestBodyPublisher);
  }

  /**
   * Get Request object with byte array
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body byte array
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   * @throws IllegalArgumentException something not supported or undefined
   * @throws NullPointerException something is null
   * @throws HttpRuntimeException Http request process exception
   */
  public static HttpRequest getRequest(
      String url, HttpMethod method, int timeout, HttpHeaders headers, byte[] requestBody)
      throws URISyntaxException, IllegalArgumentException, NullPointerException,
          HttpRuntimeException {

    if (ObjectUtil.isNull(headers)) {
      headers = HttpHeaders.NO_HEADERS;
    }

    HttpRequest.BodyPublisher requestBodyPublisher = ObjectUtil.notNull(requestBody)
            ? HttpRequest.BodyPublishers.ofByteArray(requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(new URI(url), method, timeout, headers.getHeaderList(), requestBodyPublisher);
  }

  /**
   * Get Response with String body
   *
   * @param req HttpRequest object
   * @param proxy Http proxy, Not null is enable proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   */
  public static HttpResponse<String> getResponseString(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    try {
      return getResponseRaw(req, BodyHandlers.ofString(), proxy);
    } catch (IOException | InterruptedException ex) {
      throw new HttpClientException(ex.getMessage());
    }
  }

  /**
   * Get Response with byte array body
   *
   * @param req HttpRequest object
   * @param proxy Http proxy, Not null is enable proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   */
  public static HttpResponse<byte[]> getResponseByteArray(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    try {
      return getResponseRaw(req, BodyHandlers.ofByteArray(), proxy);
    } catch (IOException | InterruptedException ex) {
      throw new HttpClientException(ex.getMessage());
    }
  }

  /**
   * Get Response with inputstream body
   *
   * @param req HttpRequest object
   * @param proxy Http proxy, Not null is enable proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   */
  public static HttpResponse<InputStream> getResponseInputStream(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException {
    try {
      return getResponseRaw(req, BodyHandlers.ofInputStream(), proxy);
    } catch (IOException | InterruptedException ex) {
      throw new HttpClientException(ex.getMessage());
    }
  }

  /**
   * send http request
   *
   * @param url URL
   * @param method request http method
   * @param timeout request timeout
   * @param headers request headers, if none is null
   * @param requestBody request body, is none is null
   * @param responseType response type, default is byte array
   * @param proxy request http proxy, if had not is null
   * @param <R> request type (String, InputStream and byte[])
   * @param <T> response type (String, InputStream and byte[])
   * @return http response
   * @throws URISyntaxException URL is an illegal address
   * @throws HttpClientException get response exception
   */
  public static <R, T> HttpResponse<?> sendRequest(String url, HttpMethod method, int timeout,
      HttpHeaders headers, R requestBody, Class<T> responseType, InetSocketAddress proxy)
      throws URISyntaxException, HttpClientException, HttpRuntimeException {

    if (headers == null) {
      headers = HttpHeaders.NO_HEADERS;
    }

    HttpRequest request = getRequestRaw(new URI(url), method, timeout, headers.getHeaderList(), getBodyPublisher(requestBody));

    try {
      return getResponseRaw(request, getBodyHandler(responseType), proxy);
    } catch (IOException | InterruptedException ex) {
      throw new HttpClientException(ex.getMessage());
    }

  }

  private static HttpRequest getRequestRaw(URI uri, HttpMethod method, int timeout,
      String[] headers, HttpRequest.BodyPublisher requestBody)
      throws IllegalArgumentException, IllegalStateException, HttpTimeoutException {
    HttpRequest.Builder builder = HttpRequest.newBuilder()
        .uri(uri)
        .method(method.name(), requestBody);
    if (headers.length > 0) {
      builder = builder.headers(headers);
    }
    try {
      builder = builder.timeout(Duration.ofSeconds(timeout));
    } catch (IllegalArgumentException ex) {
      throw new HttpTimeoutException("HTTP request timeout.");
    }
    return builder.build();
  }

  private static <T> HttpResponse<T> getResponseRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler, InetSocketAddress proxy)
      throws IOException, InterruptedException {
    if (ObjectUtil.isNull(proxy)) {
      return getResponseRaw(req, responseBodyHandler);
    }
    return getResponseProxyRaw(req, responseBodyHandler, proxy);
  }

  private static <T> HttpResponse<T> getResponseRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler)
      throws IOException, InterruptedException {
    return HttpClient.newHttpClient().send(req, responseBodyHandler);
  }

  private static <T> HttpResponse<T> getResponseProxyRaw(HttpRequest req,
      HttpResponse.BodyHandler<T> responseBodyHandler, InetSocketAddress proxy)
      throws IOException, InterruptedException {
    return HttpClient.newBuilder().proxy(ProxySelector.of(proxy)).build().send(req, responseBodyHandler);
  }

  private static <T> HttpRequest.BodyPublisher getBodyPublisher(T requestBody) {
    if (ObjectUtil.isNull(requestBody)) {
      return HttpRequest.BodyPublishers.noBody();
    }
    Class<?> type = requestBody.getClass();
    if (type.isAssignableFrom(CharSequence.class)) {
      return HttpRequest.BodyPublishers.ofString(requestBody.toString());
    }
    if (type.isAssignableFrom(InputStream.class)) {
      return HttpRequest.BodyPublishers.ofInputStream(() -> (InputStream) requestBody);
    }
    if (requestBody instanceof byte[]) {
      return HttpRequest.BodyPublishers.ofByteArray((byte[]) requestBody);
    }
    throw new UnsupportedOperationException("Can NOT found supported type");
  }

  private static <T> HttpResponse.BodyHandler<?> getBodyHandler(Class<T> type) {
    if (type.isAssignableFrom(String.class)) {
      return BodyHandlers.ofString();
    }
    if (type.isAssignableFrom(InputStream.class)) {
      return BodyHandlers.ofInputStream();
    }
    return BodyHandlers.ofByteArray();
  }

}
