/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.reflect;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.List;

/**
 * Reflection Util
 *
 * @author Leonard Woo
 */
public class ReflectionUtil {

  private static final Class<?>[] EMPTY_CLASS_ARRAY = new Class<?>[0];

  private static final Method[] EMPTY_METHOD_ARRAY = new Method[0];

  private static final Object[] EMPTY_OBJECT_ARRAY = new Object[0];

  private static final MethodFilter USER_DECLARED_METHODS = (method -> !method.isBridge() && !method.isSynthetic());

  /**
   * get class instance without parameter
   *
   * @param clazz target class
   * @param <T> target class type
   * @return class instance
   * @throws NoSuchMethodException not found constructor method
   * @throws IllegalAccessException can not access constructor method
   * @throws InvocationTargetException can not call constructor method
   * @throws InstantiationException can not instance
   */
  public static <T> T newInstance(Class<T> clazz)
      throws NoSuchMethodException, IllegalAccessException, InvocationTargetException,
      InstantiationException {
    return newInstance(clazz, EMPTY_CLASS_ARRAY, null);
  }

  /**
   * get class instance with parameter
   *
   * @param clazz target class
   * @param parameterTypes class constructor parameter types
   * @param initArgs class constructor parameter objects
   * @param <T> target class type
   * @return class instance
   * @throws NoSuchMethodException not found constructor method
   * @throws IllegalAccessException can not access constructor method
   * @throws InvocationTargetException can not call constructor method
   * @throws InstantiationException can not instance
   */
  public static <T> T newInstance(Class<T> clazz, Class<?>[] parameterTypes, Object[] initArgs)
      throws NoSuchMethodException, IllegalAccessException, InvocationTargetException,
          InstantiationException {
    return clazz.getDeclaredConstructor(parameterTypes).newInstance(initArgs);
  }

  /**
   * find method
   *
   * @param clazz method class
   * @param methodName method name
   * @param <T> class type
   * @return method object
   * @throws NoSuchMethodException not found method
   * @throws SecurityException access failed
   */
  public static <T> Method findMethod(Class<T> clazz, String methodName)
      throws NoSuchMethodException, SecurityException {
    return findMethod(clazz, methodName, EMPTY_CLASS_ARRAY);
  }

  /**
   * find a method with parameter types
   *
   * @param clazz method class
   * @param methodName method name
   * @param parameterTypes parameter types
   * @param <T> class type
   * @return method object
   * @throws NoSuchMethodException not found method
   * @throws SecurityException access failed
   */
  public static <T> Method findMethod(Class<T> clazz, String methodName, Class<?>... parameterTypes)
      throws NoSuchMethodException, SecurityException {
    return clazz.getDeclaredMethod(methodName, parameterTypes);
  }

  /**
   * class instance method caller
   *
   * @param t target class instance
   * @param methodName call method name
   * @param args parameter objects (can NOT use basic data types, cause nonsupport auto unboxing)
   * @param <T> target class
   * @return method return object
   * @throws ReflectiveOperationException can NOT found method or can NOT access method
   * @throws RuntimeException method or parameter exception
   */
  public static <T> Object invokeMethod(T t, String methodName, Object... args)
      throws ReflectiveOperationException, RuntimeException {
    return invokeMethod(t, methodName, getParameterTypes(args), args);
  }

  /**
   * class instance method caller
   *
   * @param t target class instance
   * @param methodName call method name
   * @param parameterTypes parameter types
   * @param args parameter objects
   * @param <T> target class
   * @return method return object
   * @throws ReflectiveOperationException can NOT found method or can NOT access method
   * @throws RuntimeException method or parameter exception
   */
  public static <T> Object invokeMethod(
      T t, String methodName, Class<?>[] parameterTypes, Object... args)
      throws ReflectiveOperationException, RuntimeException {
    Method method = findMethod(t.getClass(), methodName, parameterTypes);
    return invokeMethod(method, t, args);
  }

  /**
   * get parameters type
   *
   * @param args data type objects
   * @return get object types
   */
  public static Class<?>[] getParameterTypes(final Object[] args) {
    Class<?>[] types = new Class[args.length];
    for (int i = 0; i < args.length; i++) {
      types[i] = args[i].getClass();
    }
    return types;
  }

  /**
   * invoke method on target instance
   *
   * @param method target method
   * @param target target class instance
   * @param args method parameters
   * @return method return object (if exist)
   * @throws InvocationTargetException method throws an exception
   * @throws IllegalAccessException method can not access
   * @throws IllegalArgumentException method parameter type or size exception
   */
  public static Object invokeMethod(Method method, Object target, Object... args)
      throws InvocationTargetException, IllegalAccessException, IllegalArgumentException {
      return method.invoke(target, args);
  }

  /**
   * invoke method without parameter
   *
   * @param method target method
   * @param target target class instance
   * @return method return object (if exist)
   * @throws InvocationTargetException method throws an exception
   * @throws IllegalAccessException method can not access
   * @throws IllegalArgumentException method parameter type or size exception
   */
  public static Object invokeMethod(Method method, Object target)
      throws InvocationTargetException, IllegalAccessException, IllegalArgumentException {
    return invokeMethod(method, target, EMPTY_OBJECT_ARRAY);
  }

  /**
   * get methods
   *
   * @param clazz target class
   * @return methods
   * @throws SecurityException can not access method or some else
   */
  public static Method[] getDeclaredMethods(Class<?> clazz) throws SecurityException {
    Method[] declaredMethods = clazz.getDeclaredMethods();
    return declaredMethods.length == 0? declaredMethods: declaredMethods.clone();
  }

  /**
   * get methods
   *
   * @param clazz target class
   * @return methods
   * @throws SecurityException can not access method or some else
   * @throws IllegalAccessException method can not access
   */
  public static Method[] getAllDeclaredMethods(Class<?> clazz)
      throws SecurityException, IllegalAccessException {
    final List<Method> methods = new ArrayList<>(32);
    doWithMethods(clazz, methods::add);
    return methods.toArray(EMPTY_METHOD_ARRAY);
  }

  private static void doWithMethods(Class<?> clazz, MethodCallback mc)
      throws SecurityException, IllegalAccessException {
    doWithMethods(clazz, mc, null);
  }

  private static void doWithMethods(Class<?> clazz, MethodCallback mc, MethodFilter mf)
      throws SecurityException, IllegalAccessException {
    Method[] methods = getDeclaredMethods(clazz);
    for (Method method : methods) {
      if (mf != null && !mf.matches(method)) {
        continue;
      }
      mc.doWith(method);
    }
    if (clazz.getSuperclass() != null && (mf != USER_DECLARED_METHODS || clazz.getSuperclass() != Object.class)) {
      doWithMethods(clazz.getSuperclass(), mc, mf);
    } else if (clazz.isInterface()) {
      for (Class<?> superIfc : clazz.getInterfaces()) {
        doWithMethods(superIfc, mc, mf);
      }
    }
  }

  /**
   * find field with type
   *
   * @param clazz target class
   * @param name field name
   * @param type field type
   * @return field object
   * @throws NoSuchFieldException not found field
   */
  public static Field findField(Class<?> clazz, String name, Class<?> type)
      throws NoSuchFieldException {
    Class<?> searchType = clazz;
    while (Object.class != searchType && searchType != null) {
      Field[] fields = getDeclaredFields(searchType);
      for (Field field : fields) {
        if ((name == null || name.equals(field.getName())) &&
            (type == null || type.equals(field.getType()))) {
          return field;
        }
      }
      searchType = searchType.getSuperclass();
    }
    throw new NoSuchFieldException();
  }

  /**
   * get all fields
   *
   * @param clazz target class
   * @return fields
   * @throws SecurityException field security
   */
  public static Field[] getDeclaredFields(Class<?> clazz) throws SecurityException {
    return clazz.getDeclaredFields();
  }

  /**
   * get field value
   *
   * @param field target class field
   * @param target field name
   * @param <T> field name type
   * @return target field value
   * @throws IllegalArgumentException field not exist
   * @throws IllegalAccessException field access failed
   */
  public static <T> Object getField(Field field, T target)
    throws IllegalArgumentException, IllegalAccessException {
    return field.get(target);
  }

  /**
   * set field value
   *
   * @param field target class field
   * @param target field name
   * @param value field value
   * @param <T> field name type
   * @param <V> field value type
   * @throws IllegalArgumentException field not exist
   * @throws IllegalAccessException field access failed
   */
  public static <T, V> void setField(Field field, T target, V value)
      throws IllegalArgumentException, IllegalAccessException {
    field.set(target, value);
  }

//  public static void doWithFields(Class<?> clazz, FieldCallback fc) throws IllegalAccessException {
//    doWithFields(clazz, fc, null);
//  }
//
//  public static void doWithFields(Class<?> clazz, FieldCallback fc, FieldFilter ff)
//      throws IllegalAccessException {
//    Class<?> targetClass = clazz;
//    do {
//      Field[] fields = getDeclaredFields(targetClass);
//      for (Field field : fields) {
//        if (ff != null && !ff.matches(field)) {
//          continue;
//        }
//        fc.doWith(field);
//      }
//      targetClass = targetClass.getSuperclass();
//    }
//    while (targetClass != null && targetClass != Object.class);
//  }

  /**
   * get method parameters
   *
   * @param method target method object
   * @return parameters
   */
  public static Parameter[] getParameters(Method method) {
    return method.getParameters();
  }

  /**
   * doMethod filter
   */
  @FunctionalInterface
  private interface MethodFilter {

    /**
     * Determine whether the given method matches.
     * @param method the method to check
     */
    boolean matches(Method method);

    /**
     * Create a composite filter based on this filter <em>and</em> the provided
     * filter.
     * <p>If this filter does not match, the next filter will not be applied.
     * @param next the next {@code MethodFilter}
     * @return a composite {@code MethodFilter}
     */
    default MethodFilter and(MethodFilter next) {
      return method -> matches(method) && next.matches(method);
    }
  }

  /**
   * doMethod callback
   */
  @FunctionalInterface
  private interface MethodCallback {

    /**
     * Perform an operation using the given method.
     * @param method the method to operate on
     */
    void doWith(Method method) throws IllegalArgumentException, IllegalAccessException;
  }

//  @FunctionalInterface
//  public interface FieldCallback {
//
//    /**
//     * Perform an operation using the given field.
//     * @param field the field to operate on
//     */
//    void doWith(Field field) throws IllegalArgumentException, IllegalAccessException;
//  }
//
//  @FunctionalInterface
//  public interface FieldFilter {
//
//    /**
//     * Determine whether the given field matches.
//     * @param field the field to check
//     */
//    boolean matches(Field field);
//
//    /**
//     * Create a composite filter based on this filter <em>and</em> the provided
//     * filter.
//     * <p>If this filter does not match, the next filter will not be applied.
//     * @param next the next {@code FieldFilter}
//     * @return a composite {@code FieldFilter}
//     * @since 5.3.2
//     */
//    default FieldFilter and(FieldFilter next) {
//      return field -> matches(field) && next.matches(field);
//    }
//  }

}
