/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.image;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Objects;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageOutputStream;

/**
 * Image Util
 *
 * @author Leonard Woo
 */
public class ImageUtil {

  private ImageUtil(){}

  /**
   * Compress image.
   *
   * @param image a {@link BufferedImage}.
   * @param formatName a {@code String} containing the informal name of a format.
   * @param quality a {@code float} between {@code 0} and {@code 1} indicating the desired quality level.
   * @return the image byte array.
   * @throws IOException I/O Error.
   * @throws NullPointerException Image is {@code null}.
   * @throws IllegalArgumentException format name is {@code null}.
   * @see ImageIO#getImageWritersByFormatName(String)
   * @see ImageWriter
   */
  public static byte[] compressor(BufferedImage image, String formatName, float quality)
      throws IOException, NullPointerException, IllegalArgumentException {
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ImageOutputStream ios = ImageIO.createImageOutputStream(baos)) {
      Objects.requireNonNull(image);

      ImageWriter writer = ImageIO.getImageWritersByFormatName(formatName).next();
      writer.setOutput(ios);

      ImageWriteParam param = writer.getDefaultWriteParam();
      param.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
      param.setCompressionQuality(quality);
      param.setProgressiveMode(ImageWriteParam.MODE_DEFAULT);

      writer.write(null, new IIOImage(image, null, null), param);
      writer.dispose();
      return baos.toByteArray();
    }
  }

  /**
   * Write image to byte array.
   *
   * @param image a {@code BufferedImage}.
   * @param formatName a {@code String} containing the informal name of a format.
   * @return the image byte array. {@code null} if no appropriate writer is found.
   * @throws IOException I/O Error.
   * @throws NullPointerException Image is {@code null}.
   */
  public static byte[] writeImage(BufferedImage image, String formatName)
      throws IOException, NullPointerException {
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
      Objects.requireNonNull(image);
      if (ImageIO.write(image, formatName, baos)) {
        return baos.toByteArray();
      }
      return null;
    }
  }
}
