/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.jdbc;

import java.lang.reflect.Field;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import org.seppiko.commons.utils.reflect.ReflectionUtil;

/**
 * JDBC ResultSet Util.
 *
 * @author Leonard Woo
 */
public class ResultSetUtil {

  private ResultSetUtil() {}

  /**
   * Convert ResultSet to {@code 'List<Map<ColumnName, ColumnValue>>'}.
   *
   * @param resultSet ResultSet instance. Without close.
   * @return List of Map, value type is sql type.
   * @throws SQLException if a database access error occurs or this method is called on a closed
   *     result set.
   */
  public static ArrayList<HashMap<String, Object>> convert(ResultSet resultSet)
      throws SQLException {
    ArrayList<HashMap<String, Object>> result = new ArrayList<>();
    ResultSetMetaData row = resultSet.getMetaData();
    int colSize = row.getColumnCount();
    while (resultSet.next()) {
      HashMap<String, Object> rowMap = new HashMap<>();
      for (int i = 0; i < colSize; i++) {
        rowMap.put(row.getColumnName(i), resultSet.getObject(i));
      }
      result.add(rowMap);
    }
    return result;
  }

  /**
   * Convert ResultSet to entity list.
   *
   * @param resultSet ResultSet instance. Without close.
   * @param clazz entity class.
   * @return entity list.
   * @param <T> entity class type.
   * @throws SQLException if a database access error occurs or this method is called on a closed
   *     result set.
   * @throws IllegalAccessException target entity create failed.
   */
  public static <T> ArrayList<T> convert(ResultSet resultSet, Class<T> clazz)
      throws SQLException, IllegalAccessException {
    ArrayList<T> result = new ArrayList<>();
    while (resultSet.next()) {
      Field[] fields;
      try {
        fields = ReflectionUtil.getDeclaredFields(clazz);
      } catch (SecurityException e) {
        try {
          fields = ReflectionUtil.getRecordFields(clazz);
        } catch (NoSuchFieldException | SecurityException ex) {
          throw new IllegalAccessException("Failed to get entity class field.");
        }
      }

      Class<?>[] paramTypes = new Class[fields.length];
      Object[] args = new Object[fields.length];
      for (int i = 0; i < fields.length; i++) {
        fields[i].setAccessible(true);
        paramTypes[i] = fields[i].getType();
        args[i] = resultSet.getObject(i + 1);
      }

      T t;
      try {
        t = ReflectionUtil.setRecordAllFields(clazz, paramTypes, args);
      } catch (ReflectiveOperationException | IllegalArgumentException e) {
        try {
          t = ReflectionUtil.newInstance(clazz);
          for (Field field : fields) {
            ReflectionUtil.setField(field, t, field.get(t));
          }
        } catch (ReflectiveOperationException ex) {
          throw new IllegalAccessException("Call parameterless constructor failed.");
        }
      }

      result.add(t);
    }

    return result;
  }

  /**
   * Close ResultSet
   *
   * @param resultSet ResultSet instance.
   * @return true if ResultSet is closed. false is otherwise.
   * @throws SQLException if a database access error occurs.
   */
  public static boolean close(ResultSet resultSet) throws SQLException {
    if (resultSet != null && !resultSet.isClosed()) {
      resultSet.close();
      return true;
    }
    return false;
  }
}
