/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.reflect;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Objects;

/**
 * Type util
 *
 * @author Leonard Woo
 */
public class TypeUtil {

  private TypeUtil(){}

  /**
   * get Object type
   *
   * @param obj Object
   * @return object type class
   */
  public static Class<?> checkType(Object obj) {
    if (Objects.isNull(obj)) {
      return null;
    }
    if (obj.getClass().isPrimitive()) {
      if (obj instanceof Boolean) {
        return Boolean.TYPE;
      }
      if (obj instanceof Byte) {
        return Byte.TYPE;
      }
      if (obj instanceof Short) {
        return Short.TYPE;
      }
      if (obj instanceof Integer) {
        return Integer.TYPE;
      }
      if (obj instanceof Long) {
        return Long.TYPE;
      }
      if (obj instanceof Float) {
        return Float.TYPE;
      }
      if (obj instanceof Double) {
        return Double.TYPE;
      }
      if (obj instanceof Character) {
        return Character.TYPE;
      }
    }
    throw new IllegalArgumentException("Not found type");
  }

  /**
   * get type boxing class
   *
   * @param type type class
   * @return boxing class
   */
  public static Class<?> wrapper(Class<?> type) {
    if (Objects.isNull(type)) {
      return null;
    }
    if (type.isPrimitive()) {
      if (boolean.class == type) {
        return Boolean.class;
      }
      if (byte.class == type) {
        return Byte.class;
      }
      if (short.class == type) {
        return Short.class;
      }
      if (int.class == type) {
        return Integer.class;
      }
      if (long.class == type) {
        return Long.class;
      }
      if (float.class == type) {
        return Float.class;
      }
      if (double.class == type) {
        return Double.class;
      }
      if (char.class == type) {
        return Character.class;
      }
    }
    return type;
  }

  /**
   * Parser number type.
   *
   * @param number number.
   * @return number type and value.
   */
  public static TypeAndValue<?> parserNumberType(Number number) {
    if (number instanceof BigInteger) {
      return new TypeAndValue<>(BigInteger.class, (BigInteger) number);
    }
    if (number instanceof BigDecimal) {
      return new TypeAndValue<>(BigDecimal.class, (BigDecimal) number);
    }
    if (number instanceof Byte) {
      return new TypeAndValue<>(Byte.class, number.byteValue());
    }
    if (number instanceof Short) {
      return new TypeAndValue<>(Short.class, number.shortValue());
    }
    if (number instanceof Integer) {
      return new TypeAndValue<>(Integer.class, number.intValue());
    }
    if (number instanceof Long) {
      return new TypeAndValue<>(Long.class, number.longValue());
    }
    if (number instanceof Float) {
      return new TypeAndValue<>(Float.class, number.floatValue());
    }
    if (number instanceof Double) {
      return new TypeAndValue<>(Double.class, number.doubleValue());
    }
    throw new IllegalArgumentException("Not found number " + number + " of type " + number.getClass());
  }

  /**
   * Convert Object to Class type.
   *
   * @param obj object.
   * @param type target class type.
   * @param <T> type.
   * @return Class type object.
   * @throws NullPointerException sql object is null.
   * @throws IllegalArgumentException target type mismatch.
   */
  @SuppressWarnings("unchecked")
  public static <T> T convertObject(Object obj, Class<T> type)
      throws NullPointerException, IllegalArgumentException {
    if (Objects.isNull(obj)) {
      throw new NullPointerException();
    }
    if (obj.getClass().equals(type)) {
      return (T) obj;
    }
    throw new IllegalArgumentException();
  }
}
