/*
 * Copyright 2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.io.Serial;
import java.io.Serializable;
import java.util.Comparator;
import java.util.Objects;
import org.seppiko.commons.utils.internal.VersionComparator;

/**
 * Semantic Version Utility
 *
 * @see <a href="https://semver.org/">Semantic Versioning</a>
 * @author Leonard Woo
 */
public class VersionUtil implements Serializable {

  @Serial
  private static final long serialVersionUID = -1334772796709558040L;

  /** version string */
  private final String version;

  /** version major */
  private final int major;

  /** version minor */
  private final int minor;

  /** version patch */
  private final int patch;

  /** version qualifier */
  private final String qualifier;

  private VersionUtil(final String versionString) {
    if (StringUtil.isNullOrEmpty(versionString)) {
      throw new IllegalArgumentException("Version string must be not null.");
    }

    int major = 0;
    int minor = 0;
    int patch = 0;
    String qualify = "";

    int length = versionString.length();
    int type = 0;
    int val = 0;
    for (int offset = 0; offset < length; offset++) {
      char c = versionString.charAt(offset);
      if (c < CharUtil.DIGIT_ZERO || c > CharUtil.DIGIT_NINE) {
        switch (type) {
          case 0 -> major = val;
          case 1 -> minor = val;
          case 2 -> {
            patch = val;
            qualify = versionString.substring(offset + 1);
            offset = length;
          }
        }
        type++;
        val = 0;
      } else {
        val = val * 10 + c - CharUtil.DIGIT_ZERO;
      }
    }
    if (type == 2) {
      patch = val;
    }

    this.major = major;
    this.minor = minor;
    this.patch = patch;
    this.qualifier = qualify;
    this.version = versionString;
  }

  private VersionUtil(int major, int minor, int patch, String qualifier) {
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    this.qualifier = qualifier;
    this.version = generator(major, minor, patch, qualifier);
  }

  private String generator(int major, int minor, int patch, String qualify) {
    String ver = String.format("%d.%d.%d", major, minor, patch);
    if (StringUtil.hasText(qualify)) {
      ver = String.format("%s-%s", ver, qualify);
    }
    return ver;
  }

  /**
   * parser version string
   *
   * @param versionString version string
   * @return Version object
   */
  public static VersionUtil parser(String versionString) {
    return new VersionUtil(versionString);
  }

  /**
   * get version string
   *
   * @return version string
   */
  public String versionString() {
    return version;
  }

  /**
   * get major version
   *
   * @return major number
   */
  public int major() {
    return major;
  }

  /**
   * get minor version
   *
   * @return minor number
   */
  public int minor() {
    return minor;
  }

  /**
   * get patch version
   *
   * @return path number
   */
  public int patch() {
    return patch;
  }

  /**
   * get version qualifier
   *
   * @return qualifier string
   */
  public String qualifier() {
    return qualifier;
  }

  /**
   * compare
   *
   * @param other Another {@link VersionUtil}
   * @return If equal return 0, if this > other return 1, if other > this return -1.
   */
  public int compareTo(VersionUtil other) {
    if (null == other) {
      throw new NullPointerException("`other` must be not null.");
    }
    return new VersionComparator().compare(this, other);
  }

  /**
   * Check version object
   *
   * @param o version object
   * @return true, if object is VersionUtil and equals
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    VersionUtil that = (VersionUtil) o;
    return version.equals(that.versionString()) ||
        ( major == that.major() && minor == that.minor() && patch == that.patch() &&
            Objects.equals(qualifier, that.qualifier()) );
  }

  /** VersionUtil object hashcode */
  @Override
  public int hashCode() {
    return Objects.hash(version, major, minor, patch, qualifier);
  }

  /** Version string */
  @Override
  public String toString() {
    return version;
  }
}
