/*
 * Copyright 2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;


/**
 * Assertion
 *
 * @author Leonard Woo
 */
public class Assert {

  private Assert() {}

  /**
   * if not true throw message
   *
   * @param expression boolean expression.
   * @param message message.
   * @throws IllegalArgumentException if expression is false throw this.
   */
  public static void isTrue(boolean expression, String message) {
    if (!expression) {
      throw new IllegalArgumentException(message);
    }
  }

  /**
   * if true throw message
   *
   * @param expression boolean expression.
   * @param message message.
   * @throws IllegalArgumentException if expression is true throw this.
   */
  public static void isFalse(boolean expression, String message) {
    if (expression) {
      throw new IllegalArgumentException(message);
    }
  }

  /**
   * Assert that expected and actual are equal.
   *
   * @param expected expected value.
   * @param actual the value to check against {@code expected}.
   */
  public static void equals(Object expected, Object actual) {
    equals(expected, actual, null);
  }

  /**
   * Assert that expected and actual are equal.
   *
   * @param expected expected value.
   * @param actual the value to check against {@code expected}.
   * @param message the identifying message for the {@link AssertionError} ({@code null} okay).
   */
  public static void equals(Object expected, Object actual, String message) {
    if (!expected.equals(actual)) {
      throwAssertionError(message, null);
    }
  }

  /**
   * Constructs an {@link AssertionError} with a message and a cause but without expected/actual values.
   *
   * @param message the detail message; null or blank will be converted to the empty String.
   * @param cause the cause of the failure.
   * @throws AssertionError always.
   */
  static void throwAssertionError(String message, Throwable cause) {
    throw new AssertionError(message, cause);
  }

  /**
   * if string text has not any char without whitespace throw message
   *
   * @param text string text.
   * @param message message.
   * @throws NullPointerException if text have not any char throw this.
   */
  public static void hasText(String text, String message) {
    if (StringUtil.nonText(text)) {
      throw new NullPointerException(message);
    }
  }

  /**
   * Throw Exception to RuntimeException.
   *
   * <pre><code>
   *  try {
   *    throw new Exception("test");
   *  } catch (Throwable t) {
   *    throw Assert.sneakyThrow(t);
   *  }
   * </code></pre>
   *
   * @param t any exception.
   * @return {@link RuntimeException}.
   * @throws NullPointerException if t is {@code null} throw this.
   */
  public static RuntimeException sneakyThrow(Throwable t) {
    if (t == null) {
      throw new NullPointerException();
    }
    return sneakyThrow0(t);
  }

  @SuppressWarnings("unchecked")
  private static <T extends Throwable> T sneakyThrow0(Throwable t) throws T {
    throw (T) t;
  }
}
