/*
 * Copyright 2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.concurrent;

import java.util.concurrent.TimeUnit;
import org.seppiko.commons.utils.StringUtil;

/**
 * Retry task
 *
 * @author Leonard Woo
 */
public class RetryableTask {

  private RetryableTask() {}

  /**
   * Retry task caller
   *
   * <p>If {@code <T>} is Boolean than true, or CharSequence than not null or empty, or Object than not
   * null break retry loop.
   *
   * <p>caller will ignore runtime exception.
   *
   * @param callable task retry callable
   * @param times retry times
   * @param delay retry delay
   * @param unit delay time util
   * @return task callable result, failure is {@code null}
   * @param <T> task return type
   */
  public static <T> T caller(RetryCallable<T> callable, int times, long delay, TimeUnit unit) {
    T result = null;
    int i = 0;
    do {
      try {
        result = callable.call();
      } catch (RuntimeException ignored) {
        // ignored call runtime exception
      }

      // break logic
      if (result instanceof Boolean) {
        if (result.equals(Boolean.TRUE)) {
          break;
        }
      } else if (result instanceof CharSequence) {
        if (!StringUtil.isNullOrEmpty((CharSequence) result)) {
          break;
        }
      }

      // default break logic
      if (result != null) {
        break;
      }

      // do not delay last time
      if (i < (times - 1)) {
        try {
          unit.sleep(delay);
        } catch (InterruptedException ignored) {}
      }

      i++;
    } while (i < times);
    return result;
  }

}
