/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * Character Util
 *
 * @author Leonard Woo
 */
public class CharUtil {

  /**
   * NUL or '\0'
   */
  public static final Character NULL = 0x000;

  /**
   * HT or '\t'
   */
  public static final Character HORIZONTAL_TABULATION = 0x009;

  /**
   * LF or '\n'
   */
  public static final Character LINE_FEED = 0x00A;

  /**
   * VT or '\v'
   */
  public static final Character VERTICAL_TABULATION = 0x00B;

  /**
   * FF or '\f'
   */
  public static final Character FORM_FEED = 0x00C;

  /**
   * CR or '\r'
   */
  public static final Character CARRIAGE_RETURN = 0x00D;

  /**
   * Charset decode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data byte array
   * @return character buffer object
   * @see Charset
   * @see StandardCharsets
   */
  public static CharBuffer charsetDecode(Charset charset, byte[] data) {
    try {
      return charset.newDecoder().decode(ByteBuffer.wrap(data));
    } catch (CharacterCodingException ignored) {
    }
    return null;
  }

  /**
   * Charset encode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data character buffer
   * @return byte array
   * @see Charset
   * @see StandardCharsets
   */
  public static byte[] charsetEncode(Charset charset, CharBuffer data) {
    try {
      return charset.newEncoder().encode(data).array();
    } catch (CharacterCodingException ignored) {
    }
    return null;
  }

}
