/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.util.Objects;

/**
 * Version number parser util
 *
 * @author Leonard Woo
 */
public class VersionUtil {

  private final String version;
  private final int major;
  private final int minor;
  private final int patch;
  private final String qualifier;

  private VersionUtil(String versionString) {
    this.version = versionString;
    int major = 0;
    int minor = 0;
    int patch = 0;
    String qualif = "";

    int length = version.length();
    int type = 0;
    int val = 0;
    for (int offset = 0; offset < length; offset++) {
      char car = version.charAt(offset);
      if (car < '0' || car > '9') {
        switch (type) {
          case 0 -> major = val;
          case 1 -> minor = val;
          case 2 -> {
            patch = val;
            qualif = version.substring(offset);
            offset = length;
          }
          default -> {
          }
        }
        type++;
        val = 0;
      } else {
        val = val * 10 + car - 48;
      }
    }

    if (type == 2) {
      patch = val;
    }
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    this.qualifier = qualif;
  }

  private VersionUtil(int major, int minor, int patch, String qualifier) {
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    this.qualifier = qualifier;

    String ver = major + "." + minor + "." + patch;
    if (StringUtil.hasText(qualifier)) {
      ver += ver + "." + qualifier;
    }
    this.version = ver;
  }

  /**
   * parser version string
   * @param versionString version string
   * @return Version object
   */
  public static VersionUtil parser(String versionString) {
    return new VersionUtil(versionString);
  }

  /**
   * get version string
   * @return version string
   */
  public String versionString() {
    return version;
  }

  /**
   * get major version
   * @return major number
   */
  public int major() {
    return major;
  }

  /**
   * get minor version
   * @return minor number
   */
  public int minor() {
    return minor;
  }

  /**
   * get patch version
   * @return path number
   */
  public int patch() {
    return patch;
  }

  /**
   * get version qualifier
   * @return qualifier string
   */
  public String qualifier() {
    return qualifier;
  }

  /**
   * version equal
   *
   * @param major major number
   * @param minor minor number
   * @param patch patch number
   * @return true, if version is equal, false is otherwise
   */
  public boolean equals(int major, int minor, int patch) {
    return this.equals(new VersionUtil(major, minor, patch, ""));
  }

  /**
   * Check version object
   *
   * @param o version object
   * @return true, if object is VersionUtil and equals
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    VersionUtil that = (VersionUtil) o;
    return major == that.major && minor == that.minor && patch == that.patch && version.equals(
        that.version) && Objects.equals(qualifier, that.qualifier);
  }

  /**
   * VersionUtil object hashcode
   */
  @Override
  public int hashCode() {
    return Objects.hash(version, major, minor, patch, qualifier);
  }
}
