/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.net.URI;
import java.net.http.HttpClient.Version;
import java.net.http.HttpHeaders;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import javax.net.ssl.SSLSession;

/**
 * HttpResponse implementation
 *
 * @author Leonard Woo
 */
public class HttpResponseImpl<T> implements HttpResponse<T> {

  private CompletableFuture<HttpResponse<T>> resp;

  /**
   * Constructor
   *
   * @param resp http response
   */
  public HttpResponseImpl (CompletableFuture<HttpResponse<T>> resp) {
    this.resp = resp;
  }

  /**
   * Returns the status code for this response.
   *
   * @return the response code
   */
  @Override
  public int statusCode() {
    try {
      return resp.thenApply(HttpResponse::statusCode).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return -1;
    }
  }

  /**
   * Returns the HttpRequest corresponding to this response.
   *
   * @return the request
   * @see #previousResponse()
   */
  @Override
  public HttpRequest request() {
    try {
      return resp.thenApply(HttpResponse::request).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return null;
    }
  }

  /**
   * Returns an {@code Optional} containing the previous intermediate response
   * if one was received.
   *
   * @return an Optional containing the HttpResponse, if any.
   */
  @Override
  public Optional<HttpResponse<T>> previousResponse() {
    try {
      return resp.thenApply(HttpResponse::previousResponse).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return Optional.empty();
    }
  }

  /**
   * Returns the received response headers.
   *
   * @return the response headers
   */
  @Override
  public HttpHeaders headers() {
    try {
      return resp.thenApply(HttpResponse::headers).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return null;
    }
  }

  /**
   * Returns the body.
   *
   * <p> If this {@code HttpResponse} was returned from an invocation of
   * {@link #previousResponse()} then this method returns {@code null}
   *
   * @return the body
   */
  @Override
  public T body() {
    try {
      return resp.thenApply(HttpResponse::body).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return null;
    }
  }

  /**
   * Returns an {@link Optional} containing the {@link SSLSession} in effect
   * for this response.
   *
   * @return an {@code Optional} containing the {@code SSLSession} associated
   *         with the response
   */
  @Override
  public Optional<SSLSession> sslSession() {
    try {
      return resp.thenApply(HttpResponse::sslSession).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return Optional.empty();
    }
  }

  /**
   * Returns the {@code URI} that the response was received from. This may be
   * different from the request {@code URI} if redirection occurred.
   *
   * @return the URI of the response
   */
  @Override
  public URI uri() {
    try {
      return resp.thenApply(HttpResponse::uri).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return null;
    }
  }

  /**
   * Returns the HTTP protocol version that was used for this response.
   *
   * @return HTTP protocol version
   */
  @Override
  public Version version() {
    try {
      return resp.thenApply(HttpResponse::version).get();
    } catch (ExecutionException e) {
      throw new HttpRuntimeException(e.getMessage());
    } catch (InterruptedException e) {
      interruped(e);
      return Version.HTTP_1_1;
    }
  }

  private void interruped(InterruptedException ex) {
    Thread.currentThread().interrupt();
    throw new HttpInterruptedException(ex);
  }
}
