/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.net.URI;

/**
 * URI Util
 *
 * [scheme ":"] ["//" authority] "/" path ["?" query] ["#" fragment]
 * authority = [userinfo "@"] host [":" port]
 * userinfo = username [":" password]
 *
 * But the URI definition requires that the scheme must be used.
 *
 * @author Leonard Woo
 */
public class URIUtil {

  /**
   * get URI object
   *
   * @param uri URI address (like url)
   * @return URI object
   * @throws NullPointerException If uri string is null
   * @throws IllegalArgumentException If the given string violates RFC 2396
   */
  public static URI getUri(String uri) throws NullPointerException, IllegalArgumentException {
    return URI.create(uri);
  }

  /**
   * Get last path name (e.g. filename)
   *
   * @param uri URI
   * @return last path content e.g. filename
   * @throws IllegalArgumentException uri is not legal URI
   * @throws NullPointerException uri is null
   */
  public static String getLastPathname(String uri) throws IllegalArgumentException, NullPointerException {
    String filename = getUri(uri).getPath();
    return filename.substring(filename.lastIndexOf('/') + 1);
  }

}
